package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.40.2 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.40.2) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M35J0``,````$````__\``+@`````````0```````````````````````````
M````````````````````@`````X?N@X`M`G-(;@!3,TA5&AI<R!P<F]G<F%M
M(&-A;FYO="!B92!R=6X@:6X@1$]3(&UO9&4N#0T*)`````````!010``9(8+
M`,7Y<V@``````````/``+@(+`@(K`$0```#,`````@```!`````0`````$``
M`0`````0`````@``!``````````%``(```````!@`0``!```H+8!``,``(``
M`"`````````0```````````0````````$``````````````0```````!`,\!
M````$`$`M`H````@`0!0*P```-```.`!``````````````!0`0`P`````,``
M`!P`````````````````````````````````````````````````````````
MR!(!`&`"```````````````````````````````````N=&5X=````!A#````
M$````$0````$``````````````````!@``!@+F1A=&$```"``````&`````"
M````2```````````````````0```P"YR9&%T80``?$,```!P````1````$H`
M`````````````````$```$`N8G5I;&1I9#4`````P`````(```".````````
M``````````!```!`+G!D871A``#@`0```-`````"````D```````````````
M````0```0"YX9&%T80``V`$```#@`````@```)(``````````````````$``
M`$`N8G-S`````.`!````\`````````````````````````````"```#`+F5D
M871A``#/`0`````!```"````E```````````````````0```0"YI9&%T80``
MM`H````0`0``#````)8``````````````````$```,`N<G-R8P```%`K````
M(`$`4"L```"B``````````````````!```!`+G)E;&]C```P`````%`!```"
M````S@``````````````````0```0@``````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````$B#["A(C0W%0```
MZ!`[``!%,<`QTC')Z"0[``!%,<`QTC')Z"@[``!%,<`QTC')Z"P[``!%,<`Q
MTC')2(/$*.DL.P``D)"0D)"0D)"0D)"03(T%J4\``#'22(T-$````.EK.```
M9F8N#Q^$``````##D)"0D)"0D)"0D)"0D)"0055!5%575E-(@^PH2(G-28G4
M2(7)#X2E````2(L%CJ```$R+*$V%[0^$D@````^V`83`=`0\/75/,?])BUT`
M2(7;=#%(8_],B>YF#Q]$``!)B?A(B>I(B=GH\CD``(7`=0:`/#L]=#A(BUX(
M2(/&"$B%VW7;2(G82(/$*%M>7UU!7$%=PP\?`$B)SP^V1P%(@\<!A,!T!#P]
M=?`I[^N<D$PI[DB-7#L!2,'^`TB)V$&)-"1(@\0H6UY?74%<05W#,=M(B=A(
M@\0H6UY?74%<05W#9F8N#Q^$```````/'P!64TB![+@```!(BS6@GP``2(L&
M2(F$)*@````QP$B-5"0@2(G+Z#8X``"%P'4RBU0D,('B`/```('Z`$```'0(
M@?H`H```=1JZ`@```$B)V>A[-P``A<`/E,`/ML#K`Y`QP$B+E"2H````2"L6
M=0I(@<2X````6U[#Z$(W``"0D$%505155U932('LR````$B+/1J?``!(BP=(
MB80DN````#'`2(G+2(7)=`6`.0!U*$B+A"2X````2"L'#X4F`0``2('$R```
M`%M>7UU!7$%=PP\?@`````#HXS<``$B)QDB%P'3+2(GQZ/,W``!(A<`/A+(`
M``"`>!@N2(UH&'40@'T!`'3?9BX/'X0``````(!X&"X/A*8```!(B=GH/C@`
M`$B)Z4F)Q.@S.```18UL!`)-8^U,B>GH,S<``$B);"0H38GH,=)(B5PD($F)
MQ$R-#4I=``!(B<'H4C8``$R)X4B-5"0PZ+4W``!,B>&#^/]T$(M$)$`E`/``
M`#T`0```=%WH&#@``$R)X>B`-@``2(GQZ$@W``!(A<`/A57___\/'X``````
M2(GQZ#`V``!(B=GH6#<``.GS_O__#Q\`@'T!+@^%4/___X!]`@`/A!'____I
M0?___P\?@`````#HF_[__^NAZ-0U``"0#Q\`052X,!```%575E/H,#4``$@I
MQ$B+/::=``!!N/\/``!(BP=(B80D*!```#'`2(UL)"!(B<M(B=9(C0UW7```
M2(GJZ+DV``!!B<"%P'X1@^@!2&/0@'P4(`!U187`=3E(BPU*G0``Z,4T``!)
MB?!(B=I(BPCH1S0``$B+A"0H$```2"L'=5%(@<0P$```6UY?74%<PP\?0`!!
MB<`/'T0``$UCX+HO````2(GI38G@Z-TU``!(A<!TITB+#?&<``#H;#0``$V)
MX4F)Z$B)VDB+".CC,P``ZZ+H]#0``)`/'P!!5T%6055!5%575E-(@^Q(2(L=
MR9P``$B+`TB)1"0X,<`QP(`Z/4B)U@^4P$B)ST@!QDB)\>A6-@``2(U4)#1(
MB?E(B<7H!OS__TF)Q$B%P`^$^@```$B)P>@R-@``2#GH#X/)````3&-T)#1,
MBRV-G```2<'F`P^V!SP]#X26`0``A,`/A(X!``!(B?@/'P`/ME`!2(/``832
M=`6`^CUU[T@I^$B838ME`$B-3"@"Z.PT``!-`?1)B00D2(7`=%!)BT4`2(U/
M`4J+%#`/M@>(`H3`="`\/74+ZQH/'T0``(3`=!$/M@%(@\(!2(/!`8@"/#UU
MZ\8"/3'`9@\?1```#[8,!HA,`@%(@\`!A,EU\$B+1"0X2"L##X4/`0``2(/$
M2%M>7UU!7$%=05Y!7\.0,<!F#Q]$```/MA0&08@4!$B#P`&$TG7PZ\9F#Q]$
M``!,BRVIFP``38M]`$F#/P`/A+H```!,B?A%,>0/'T0``$B#P`A$B>)!@\0!
M2(,X`'7O1(U*`TUCR4G!X0.+!3[:``"%P'5:3(G)3(E,)"C'!2C:```!````
MZ/,S``!(B<%(A<`/A%?___]-8_1,BTPD*$R)^DG!Y@--B?#H`#,``$F)10!(
MB<%!C40D`42)9"0T2)A(QP3!`````.F'_O__3(GY3(G*Z#4T``!)B44`2(G!
M2(7`#X0%____36/T2<'F`^O##Q]``#'`Z8?^__]!N1````!%,>3I7____^B^
M,@``D&9F+@\?A```````9I!32(/L,$B+#82:``#H_S$``$R-#7!<``!,C05A
M,0``2(L(2(T5?%D``$B)P^B',0``3(T-B5D``$B+"\=$)"@`````3(E,)"!,
MC04A$P``2(T5;%D``.A5,0``D$B#Q#!;PV9F+@\?A```````#Q\`2(/L*$B+
M#16:``#HD#$``$R-#0%<``!,C07R,```2(L(2(T5#5D``$B#Q"CI%S$``&9F
M+@\?A```````#Q]``#'`PV9F+@\?A```````9I!!5[C8$```059!54%455=6
M4^A*,0``2"G$2(L=P)D``$B+`TB)A"3($```,<!(C6PD0$B)SDB)UTB-#=I8
M``!(B>KH"?G__[HO````2(GQ2(G#Z`DS``!(A<!T1$B)\DB-#;Q8``#HE?S_
M_TB)\$B+-6N9``!(BY0DR!```$@K%@^%B`$``$B!Q-@0``!;7E]=05Q!74%>
M05_#9@\?1```2(GY2(T]A5@``$R-I"3`````Z,DR``!(B?I,C3UW6```2(G!
MZ/<R``!)B<5(A<!TBT&`?0``#X2"````#Q]``$B%VW0/3(GJ2(G9Z(`R``"%
MP'1-3(GIZ)0R``!)C50%_TF)QDDYU7(3ZR$/'T0``,8"`$B#Z@%).=5T!8`Z
M+W3O3(GIZ&<R``!)B<9(B?'H7#(``$F-1`8"2#W_#P``=D](B?HQR>AU,@``
M28G%2(7`#X0%____08!]``!U@DB%VW0%@#LN=!A!O@$```!,C2VB5P``Z[5F
M#Q^$``````"`>P$`=+OKX`\?A```````2(ET)#`QTDR)X4&X`!```$R)?"0H
M3(T-A5<``$R);"0@Z"HP``!(B>I,B>'HCS$``(7`#X5X____BT0D4"4`\```
M/0"````/A63___^Z`0```$R)X>@6,```A<`/A4____],B>)(C0TF5P``Z/_Z
M__],B>'H=S$``.E@_O__Z-TO``"09F8N#Q^$``````"04TB#[""Z+P```$B)
MR^A^,0``2(U0`4B%P$@/1=I(B=E(@\0@6^DV,0``9@\?1```5;@X$```5U93
MZ`(O``!(*<1(BRUXEP``2(M%`$B)A"0H$```,<!(B<Y(A<D/A+X```"`.0`/
MA+4```#H`#$``$B)PDB-0`%(/0`0```/A\T```!(C5PD($B)WX/X"'((B<'!
MZ0/S2*4QR:@$=`F+#HD/N00```"H`G0.1`^W!`YF1(D$#TB#P0*H`70'#[8$
M#H@$#TB-1!/_2#G#<@[K$<8``$B#Z`%(.=AT!8`X+W3ONB\```!(B=GHHC``
M`$B%P'0M2#G#<E!(B=GH8#```$B+E"0H$```2"M5`'5&2('$.!```%M>7UW#
M9@\?1```2(N$)"@0``!(*T4`=29(C0W"50``2('$.!```%M>7UWI&S````\?
M`,8``.NK#Q\`,<#KK.AW+@``D&8/'T0``$%7059!54%455=64TB#[#A,BSU)
ME@``2(L]8I8``$B--955``!)BP=(B40D*#'`2(U<)"1(B=I(B?'HD?7__TB%
MP'0\2&-4)"1(BP=(C0302(M0"$B)$$B%TG38#Q]``$B+4!!(@\`(2(D02(72
M=?!(B=I(B?'H5?7__TB%P'7$2(L]^94``$B--355``!(B=I(B?'H-_7__TB%
MP'1"2&-4)"1(BP=(C0302(M0"$B)$$B%TG389BX/'X0``````$B+4!!(@\`(
M2(D02(72=?!(B=I(B?'H]?3__TB%P'6^2(L]F94``$B--=U4``!(B=I(B?'H
MU_3__TB%P'1"2&-4)"1(BP=(C0302(M0"$B)$$B%TG389BX/'X0``````$B+
M4!!(@\`(2(D02(72=?!(B=I(B?'HE?3__TB%P'6^2(L].94``$B--894``!(
MB=I(B?'H=_3__TB%P'1"2&-4)"1(BP=(C0302(M0"$B)$$B%TG389BX/'X0`
M`````$B+4!!(@\`(2(D02(72=?!(B=I(B?'H-?3__TB%P'6^2(L]V90``$B-
M-2U4``!(B=I(B?'H%_3__TB%P'1"2&-4)"1(BP=(C0302(M0"$B)$$B%TG38
M9BX/'X0``````$B+4!!(@\`(2(D02(72=?!(B=I(B?'HU?/__TB%P'6^2(LM
M>90``$B-/=U3``!(B=I(B?GHM_/__TB%P'1"2&-4)"1(BT4`2(T$T$B+4`A(
MB1!(A=)TUV8/'X0``````$B+4!!(@\`(2(D02(72=?!(B=I(B?GH=?/__TB%
MP'6^2(LM&90``$B-/2M3``!(B=I(B?GH5_/__TB%P'1"2&-4)"1(BT4`2(T$
MT$B+4`A(B1!(A=)TUV8/'X0``````$B+4!!(@\`(2(D02(72=?!(B=I(B?GH
M%?/__TB%P'6^2(LMN9,``$R-)2E3``!(B=I,B>'H]_+__TB%P'1"2&-4)"1(
MBT4`2(T$T$B+4`A(B1!(A=)TUV8/'X0``````$B+4!!(@\`(2(D02(72=?!(
MB=I,B>'HM?+__TB%P'6^3(LM69,``$B-+=-2``!(B=I(B>GHE_+__TB%P'1"
M2&-4)"1)BT4`2(T$T$B+4`A(B1!(A=)TUV8/'X0``````$B+4!!(@\`(2(D0
M2(72=?!(B=I(B>GH5?+__TB%P'6^3(LU^9(``$R-+7U2``!(B=I,B>GH-_+_
M_TB%P'1"2&-4)"1)BP9(C0302(M0"$B)$$B%TG389BX/'X0``````$B+4!!(
M@\`(2(D02(72=?!(B=I,B>GH]?'__TB%P'6^3(LUF9(``$R-+;11``!(B=I,
MB>GHU_'__TB%P'1"2&-4)"1)BP9(C0302(M0"$B)$$B%TG389BX/'X0`````
M`$B+4!!(@\`(2(D02(72=?!(B=I,B>GHE?'__TB%P'6^2(G:2(T-RU$``.B!
M\?__2(7`=`M(B<)(B>GH(?7__TB)VDB-#;U1``#H8O'__TB%P'0/2(G"2(T-
MNU$``.C^]/__2(G:2(T-MU$``.@_\?__2(G"2(7`=#%(B?GHW_3__TB+1"0H
M22L'=4!(C16R40``2(GQ2(/$.%M>7UU!7$%=05Y!7^FV]/__2(G:2(T-?U$`
M`.CW\/__2(7`=,-(B<),B>'HE_3__^NVZ(`I``"09F8N#Q^$```````/'T``
M4TB#[#!(BQU4D0``2(T-Y%```$B+`TB)1"0H,<!(C50D).BI\/__2(7`="P/
MMA"`^C`/E,"$T@^4P@G0@_`!#[;`2(M4)"A(*Q-U$$B#Q#!;PV8/'T0``#'`
MZ^;H#RD``)!F9BX/'X0```````\?`%932(/L.$B+!>.0``!,B40D8$B-="18
M2(G+3(E,)&A(B50D6$B)="0@2(L02(E4)"@QTN@(*```28GP2(G:2(M(&.BI
M*@``N?\```#HWR@``)!F9BX/'X0```````\?`%6X.``!`%=64^@"*```2"G$
M2(LU>)```#'20;@"````2(L&2(F$)"@``0`QP(G-2(U<)"#H]R@``$B->/](
MB?I(P?H_2,'J,$B-!!</M\!(*=!(*<</B)D````/'T0``$4QP$B)^HGIZ,,H
M``!(@_C_#X22````0;@(``$`2(G:B>GH*2D``(/X_W1Q2)A(@_@'=DQ(C40#
M^$@YV')"2(T5ZSX``.L29@\?A```````2(/H`4@YV'(G2(L*2#D(=>](*=A(
M`?A(BY0D*``!`$@K%G5!2('$.``!`%M>7UW#2('O```!`$B!_P``__\/A6S_
M__](Q\#_____Z\I(C0V?3P``Z)?^__](C0V&3P``Z(O^___HAB<``)`/'T0`
M`$B#["CH]R<``(G!Z.`G``!(A<!T`TB+`$B#Q"C##Q\`055!5%575E-(@^Q8
M2(LM/8\``$B-/55/``!(BT4`2(E$)$@QP$B)SDB-5"1$2(GYZ(ON__](B<-(
MA<!T!8`X`'4F2(M$)$A(*T4`=7A(B?)(B?E(@\186UY?74%<05WI#/+__P\?
M0`!(B?'HD"@``$B)V4F)Q.B%*```38UL!`),B>GHB"<``$B)7"0P,=)-B>A)
MB<1(C04#3@``2(ET)"!,C0WY30``2(E$)"A,B>'HFR8``$B+1"1(2"M%`'4%
M3(GBZXOHEB8``)`/'T0``$%7059!54%455=64TB![*@!``!(BS5FC@``2(L&
M2(F$))@!``!(C06+30``28G.2(TU=4X``$B-#7Q-``!F2`]NQDB--6=.``!(
MQX0D\`````````!(QX0D2`$```````!(QX0D$`$```````!F2`]NR$B-!3Y.
M``!F#VS!9D@/;M!(C04[3@``#RF$).````!F2`]NQDB--2).``!F2`]NV&8/
M;,)(C04@3@``#Q&$)"@!``!F2`]NQDB--0=.``!F2`]NX&8/;,,/$80D.`$`
M`&9(#V[&2(UT)&!F#VS$2(GR#RF$)``!``#H`.W__TB)PTB%P'0)@#@`#X6/
M!```Z`HF``")P>CS)0``2(7`#X2*`@``3(LX387_#X1^`@``3(GYZ`8G``!,
MC6P``4R)Z4R)Z^@&)@``2(G'00^V!TB)?"18A,!T94R-)85-``!(Q\7^____
M#Q^$```````/ML!-B>@QTDB)^8E$)"!-B>%)@\<!Z`4E``!,.>M,B>I)B>A(
M#T/32(G03"GH2(/``D@YT$@/0L))*=!(@\<"30'H3HTL`$$/M@>$P'6Q2(VL
M)"`!``!(C0V:3```ZPU(BTT(2(/%"$B%R70S2(GRZ!OL__](B<-(A<!TXX`X
M`'3>2(G!Z.;L__^%P'322(G9Z"HF``!(B<-(A<!U0F:02(VL).````!(BUT`
M2(7;="^`.P`/A'8!``!(B=GHKNS__X7`#X4&`0``2(M="$B#Q0A(A=MUVF8/
M'X0``````$B)V4R-+8Y+``#HX24``$B+?"182(G%2(GYZ-$E``!$C:0%!`0`
M`$UCY$R)X>C.)```2(E\)#@QTDV)X$B)Q4B-!5Y,``!,B6PD*$R-#4E,``!(
MB40D,$B)Z4B)7"0@Z-PC``"ZP`$``$B)Z>B_)```@_C_#X26````2(GR2(GI
MZ&LD``")PX7`=12+1"1P)0#P```]`$````^$"`$``$F+'NCI(@``28GI2(T5
M/TP``$B+2!A)B=CHTR,``#'_2(LU>HL``$B+A"28`0``2"L&#X49!```2(GX
M2('$J`$``%M>7UU!7$%=05Y!7\-FD$B)V4B#Q0CHY"0``$B)PTB%P`^$P/[_
M_^GS_O__#Q\`Z/LB``"#.!$/A%S____H[2(``$F+'HLPZ&,B``!)B>E)B=A(
MC15V2P``2(M(&(ET)"#H22,``.EQ____#Q]``#';Z:G^__]F#Q^$``````!(
MC9PD``$``.L*9@\?1```2(/#"$B+"TB%R0^$"P(``$B)\N@HZO__2(7`=..`
M.`!TWDB)P>A&)```28G'2(7`=,[I._W__XM\)'CH(",``#G'#X7G_O__BT0D
M<"7_`0``/<`!```/A=/^__],B>'H+",``$B)\DB-#8%*``!(B<?HRNG__TB%
MP`^$T0$``$F+#DB)PNAF\/__28G'2(7`#X2Z`0``3(GYZ.(C``!)B<"#^`=^
M0DB82(T5)$L``$R)1"1828U,!_CHHB,``(7`=29,BT0D6$&#^`@/A'0"``!!
MC4#W2)A!@#P'+P^$8P(``&8/'T0``$B)\DB--75)``!(B?'H1NG__TB%P`^$
M70$```^V`#PP#X12`0``A,`/A$H!``!(B?%(C16T20``2(TURD@``.C$[/__
M3(TUW4H``.@((@``2(ET)#A-B>`QTHE$)#!,C0VU2@``2(GY3(EL)"A(B6PD
M(.AQ(0``ZT`/'X``````Z#,A``"#.!%U0>C)(0``@\,!2(ET)$!-B?&)7"0X
M38G@,=)(B?F)1"0P3(EL)"A(B6PD(.@O(0``NL`!``!(B?GH$B(``(/X_W2U
M2(GIZ'4A``!(B?I(C0U$2```Z";L__](B?GHKOG__^EX_?__9@\?A```````
M2(G!Z)CY__](BSWAB```2(N$))@!``!(*P</A8`!``!(B=E(@<2H`0``6UY?
M74%<05U!7D%?Z54B``"Y#0```+L-````3(T]R$@``$&]#0```.A9(0``2(E$
M)%A(B<>X4P```.E1^___#Q^``````$V+/ND^_O__#Q^$``````"Z```!`$R)
M^>AC(0``08G&A<`/A)[^__^)P>C!]___28G'2(7`#XB+_O__13'`2(U0]D2)
M\>C6(```2(V4)%H!``!!N`8```!$B?'H0"$``$B#^`8/A<L```"!O"1:`0``
M`$-!0P^%2_[__V:!O"1>`0``2$4/A3O^__]%,<!)C5?.1(GQZ(8@``!(C9PD
M8`$``$&X*````$2)\4B)VNCM(```2(/X*'5\2(E<)#!-B>`QTDB)^4B-!=1&
M``!,B6PD*$R-#<I(``!(B40D.$B);"0@QH0DB`$```#HD!\``.EN_O__38MV
M"$V%]@^$EOW__TR)\>@F(0``2(/X`P^&A/W__TF-3`;\2(T5;$@``.CK(```
MA<!-#T3^Z6C]___H6Q\``$B-#55(``#H3_;__Y!F9BX/'X0```````\?`%57
M5E-(@^PX2(L-$8<``$B+/1J'``!(BP=(B40D*#'`2(UL)"3H=AX``$B-#7I&
M``!(B<-(BP`/MI"P````B)"Q````2(GJZ%3F__](A<!T"8`X,0^$5@$``.AQ
M\/__2(L+0;D#````0;@!````2(T5#$@``.C>'0``2(G&2(7`="=(BT`02(L+
M3(T%\T<``$B+$.B8'0``2(M&$$B+$/9"#D`/A3<!``!(BPM!N0,```!!N`$`
M``!(C17)1P``Z)0=``!(B<9(A<!T*DB)ZDB-#9Q%``#HQ>7__TF)P$B%P`^$
MV0```$B+1A!(BPM(BQ#H.AT``$B+"T&X`0```$B-%81'``#H51T``$B+"TR-
M!8I'``!(B<+H$QT``$B+`TB-#4)%``!(B>K&@+$`````Z&KE__](B07KQ```
M2(G!2(7`=#NZP`$``.C1'@``C5`!@_H!=@F#^!$/A8T```!(BT0D*$@K!P^%
MGP```$B#Q#A;7E]=PV8/'X0``````$B+#:G$``#H)/?__TB)!97$``!(B<%(
MA<!UJDB-#09'``#HP1P``&8/'X0``````(!X`0`/A*7^___IF_[__Y!(BP5I
MQ```2(L02(M&$$B+".B:Y___Z1O___\/'T0``$B+"^AH'```Z;S^__]!B<%(
MBP4YQ```3(L%*L0``$B-#<M&``!(BQ#H6QP``.@N'0``D&9F+@\?A```````
M9I!75E-(@^PP2(LU`H4``$B+!DB)1"0H,<!(B<M(C50D)$B-#8!$``#H5.3_
M_TB%P'0<#[80A-(/E,"`^C`/E,((T'4*2(7;=`6`.P!U$TB+1"0H2"L&=69(
M@\0P6UY?PY!(B=GH&.W__[HO````2(G!2(G'Z&@>``!(C5`!2(7`2`]%^DB)
M^>@E'@``0;@$````2(T5W$0``$B)P>@P'@``A<!UJ4B+1"0H2"L&=0](B=E(
M@\0P6UY?Z2/E___H7AP``)"0D)"0D)"0D)"0D)"005=!5D%505155U932('L
MN`(``$B+/2:$``!(BP=(B80DJ`(``#'`2(G*3(U1.$B-3"0@3(V<)*````!)
MB<EFD$&+`DF+&DF#P1!)@\((#\B)P$F)V$F)0?!(B=A)P>@@2,'H*$C!ZS@E
M`/\``$@)V$R)PTG!X`A(P>,808'@``#_`(G;2`G83`G`28E!^$TYV76M9@]O
M1"0P9@]O7"1`2(V$)(@```!(@<%H`@``9@]O5"109@]O+?2"```/'T``\P]O
M"&8/;^)F#V]0V$B#P!!F#^]`B&8/[\IF#^_!9@]OR&8/<]`?9@]S\0%F#^O!
M9@_;Q0\I0`AF#V_#9@]OW$@YR'6[3(L:2(MR$$B+6B!(BWH83(G83(G92(MJ
M"$R+5"0H2,'I&TC!X`5("<@#1"0@3HV4%YEY@EJ-C!B9>8):2(GP2#'X2"'H
M2#'X1(T,`4B)Z4B)Z$C!Z`)(P>$>3(G+2`G!2(GP2,'K&T&)R$PQP$PAV$@Q
M\$P!T$V)RDG!X@5)"=I&C2003(G838G:2<'J`DC!X!Y-B>5,"=!,BU0D,$G!
M[1M(,<&)PTPAR4Z-E!:9>8):3#'!3`'138GB2<'B!4T)ZDR+;"0X00'*3(G)
M2<'I`DC!X1Y%B=)/C80HF7F"6DP)R4V)U4@QR$&)R4PAX$G![1M(,=A,`<!-
MB=!)P>`%30GH3(ML)$!!`<!,B>!)P>P"2,'@'D6)P$J-G"N9>8):3`G@38G%
M2#'!2<'M&T&)Q$PAT4PQR4@!V4R)PTC!XP5,">M,BVPD2`'+3(G12<'J`DC!
MX1Z)VT^-C"F9>8):3`G128G=2#'(2<'M&T&)RDPAP$PQX$P!R$F)V4G!X05-
M">E,BVPD4$$!P4R)P$G!Z`)(P>`>18G)3XVD+)EY@EI,"<!-B<U(,<%)P>T;
M08G`2"'93#'13`'A38G,2<'D!4T)[$R+;"1800',2(G92,'K`DC!X1Y%B>1/
MC90JF7F"6D@)V4V)Y4@QR$G![1N)RTPAR$PQP$P!T$V)XDG!X@5-">I,BVPD
M8$$!PDR)R$G!Z0)(P>`>18G23XV$*)EY@EI,"<A-B=5(,<%)P>T;08G!3"'A
M2#'93`'!38G02<'@!4T)Z$R+;"1H00'(3(GA2<'L`DC!X1Y%B<!*C9PKF7F"
M6DP)X4V)Q4@QR$G![1M!B<Q,(=!,,<A(`=A,B<-(P>,%3`GK3(ML)'`!PTR)
MT$G!Z@)(P>`>B=M/C8PIF7F"6DP)T$F)W4@QP4G![1M!B<),(<%,,>%,`<E)
MB=E)P>$%30GI3(ML)'A!`<E,B<%)P>@"2,'A'D6)R4^-I"R9>8):3`G!38G-
M2#'(2<'M&T&)R$@AV$PQT$P!X$V)S$G!Y`5-">Q,BZPD@````$$!Q$B)V$C!
MX!Y(P>L"18GD3XV4*IEY@EI("=A-B>5(,<%)P>T;B<-,(<E,,<%,`=%-B>))
MP>(%30GJ3(NL)(@```!!`<I,B<E)P>D"2,'A'D6)TD^-A"B9>8):3`G)38G5
M2#'(2<'M&T&)R4PAX$@QV$P!P$V)T$G!X`5-">A,BZPDD````$$!P$R)X$G!
M[`)(P>`>18G`2HV<*YEY@EI,">!-B<5(,<%)P>T;08G$3"'13#')2`'93(G#
M2,'C!4P)ZTR+K"28`````<M,B=%(P>$>2<'J`HG;3XV,*9EY@EI,"=%)B=U(
M,<A)P>T;08G*3"'`3#'@3`'(28G92<'A!4T)Z4R+K"2@````00'!3(G`2<'H
M`DC!X!Y%B<E/C:0LF7F"6DP)P$V)S4@QP4G![1M!B<!((=E,,=%,`>%-B<Q)
MP>0%30GL1HTT(4B)V4C!ZP)(P>$>2`G92(N<)*@```!(,<A!B<Q,(<A-C90:
MF7F"6DR)\TPQP$C!ZQM,`=!-B?))P>(%20G:2(N<)+````!!`<),B<A(P>`>
M2<'I`D6)TDV-A!B9>8):3`G(3(G32#'!2,'K&T&)P4PA\4PQX4D!R$R)T4C!
MX05("=E,B?-)P>X"2,'C'HG)3`GS20'(2(N,)+@```!(,=A%B<5,(=!)C8P,
MF7F"6DV)[$PQR$G![!M(`<A,B>E(P>$%3`GAB<E(`<A,B=%)P>H"2,'A'D&)
MQ$P)T4F)VHG;23'*13'03(N4),````!/C8P1H>O9;DV)XDT!P4V)X$G!ZAM)
MP>`%30G018G`30'!38GH2<'@'DG![0)%B<I(`YPDR````$T)Z$F)S8G)2`.,
M)-````!-,<5$,>A-B=5(C9P#H>O9;DR)T$G![1M(P>`%3`GHB<!(`<-,B>!)
MP>P"2,'@'D&)W4P)X$V)Q$6)P$P#A"38````23'$13'A38GL3HV,":'KV6Y,
MB>E)P>P;2,'A!4P)X8G)20')3(G12<'J`DC!X1Y%B<Q,"=%)B<*)P$DQRD$Q
MVDR)XT^-E!"AZ]EN38G@2,'K&TG!X`5)"=A%B<!-`<)-B>A)P>T"2<'@'D2)
MTTT)Z$F)S4@#A"3@````B<E-,<5(`XPDZ````$4QZ4F)W4Z-C`BAZ]EN2(G8
M2<'M&TC!X`5,">B)P$D!P4R)X$G![`)(P>`>18G-3`G@38G$18G`3`.$)/``
M``!),<1%,>)-B>Q.C901H>O9;DR)Z4G![!M(P>$%3`GAB<E)`<I(B=E(P>L"
M2,'A'D6)U$@)V4B)PXG`2`.$)/@```!(,<M!,=E,B>-/C8P(H>O9;DV)X$C!
MZQM)P>`%20G818G`30'!38GH2<'M`DG!X!Y$B<M-">A)B<V)R4TQQ4@#C"0`
M`0``13'J28G=3HV4$*'KV6Y(B=A)P>T;2,'@!4P)Z(G`20'"3(G@2<'L`DC!
MX!Y%B=5,">!-B<1%B<!,`X0D"`$``$DQQ$4QX4V)[$Z-C`FAZ]EN3(GI2<'L
M&TC!X05,">&)R4D!R4B)V4C!ZP)(P>$>18G,2`G92(G#B<!(`X0D$`$``$@Q
MRT$QVDR)XT^-E!"AZ]EN38G@2,'K&TG!X`5)"=A%B<!-`<)-B>A)P>T"2<'@
M'D2)TTT)Z$F)S8G)33'%13'I28G=3HV,"*'KV6Y(B=A(P>`%2<'M&T@#C"08
M`0``3`GHB<!)`<%,B>!)P>P"2,'@'D6)S4P)X$V)Q$6)P$P#A"0@`0``23'$
M13'B38GL3HV4$:'KV6Y,B>E)P>P;2,'A!4P)X8G)20'*2(G92,'K`DC!X1Y%
MB=1("=E(B<.)P$@#A"0H`0``2#'+03'93(GC3XV,"*'KV6Y-B>!(P>L;2<'@
M!4D)V$6)P$T!P4V)Z$G![0))P>`>1(G+30GH28G-B<E-,<5%,>I)B=U.C900
MH>O9;DB)V$G![1M(P>`%3`GH2`.,)#`!``")P$D!PDR)X$G![`)(P>`>18G5
M3`G@38G$18G`3`.$)#@!``!),<1%,>%-B>Q.C8P)H>O9;DR)Z4G![!M(P>$%
M3`GAB<E)`<E(B=E(P>L"2,'A'D6)S$@)V4B)PXG`2`.$)$`!``!(,<M!,=I,
MB>-/C900H>O9;DV)X$C!ZQM)P>`%20G818G`30'"38GH2<'M`DG!X!Y$B=--
M">A)B<V)R4TQQ44QZ4F)W4Z-C`BAZ]EN2(G82<'M&TC!X`5,">B)P$D!P4R)
MX$C!X!Y)P>P"18G-2`.,)$@!``!,">!-B<1%B<!,`X0D4`$``$DQQ$4QXDZ-
MI!&AZ]EN3(GI38GJ2<'J&TC!X05,"=&)R4D!S$B)V4C!ZP)(P>$>18GB2`G9
M2(G#B<!(`X0D6`$``$@QRT$QV4R)TT^-C`BAZ]EN38G02,'K&TG!X`5)"=A,
MB>M)P>T"2,'C'D4!P4F)R(G)3`GK18G)23'828G>08G=13'@38G,3HV$`*'K
MV6Y,B<A)P>P;2,'@!4P)X$6-)`!-B=!)P>H"N-R\&X])P>`>30G030G&18G"
M3"'332'.20G>2(N<)&`!``!(`<-(`=E,B>-,`?%-B>9(P>,%2<'N&TP)\P'+
M3(G)2<'I`DC!X1Z)VTP)R4D)R$F)SD&)R4TAX$TAUDT)QDR+A"1H`0``20'`
M30'H28G=30'P28G>2<'E!4G![AM-"?5)B<Y%`<5-B>!)P>P"2<'@'D6)[4T)
MX$6)Q$T)QDDAWDPAX4P)\4R+M"1P`0``20'&30'638GJ3`'Q38GN2<'B!4G!
M[AM-"?)!`<I(B=E(P>L"2,'A'D6)TD@)V4D)R$F)SHG+32'H32'F30G&3(N$
M)'@!``!)`<!-`<A-B=%-`?!-B=9)P>$%2<'N&TT)\4F)SD4!P4V)Z$G![0))
MP>`>18G)30GH18G%30G&32'63"'I3`GQ3(NT)(`!``!)`<9-`>9-B<Q,`?%-
MB<Y)P>0%2<'N&TT)]$$!S$R)T4G!Z@)(P>$>18GD3`G120G(28G.08G*32'(
M32'N30G&3(N$)(@!``!)`<!)`=A,B>--`?!-B>9(P>,%2<'N&TP)\TF)SD0!
MPTV)R$G!Z0))P>`>B=M-"<A%B<%-"<9-(>9,(<E,"?%,B[0DD`$``$D!QDT!
M[DF)W4P!\4F)WDG!Y05)P>X;30GU00'-3(GA2,'A'DG![`)%B>U,">%)"<A)
MB<Y!B<Q)(=A-(<Y-"<9,BX0DF`$``$D!P$T!T$V)ZDT!\$V)[DG!X@5)P>X;
M30GR28G.10'"28G82,'K`DG!X!Y%B=))"=A$B<--"<9-(>Y((=E,"?%,B[0D
MH`$``$D!QDT!SDV)T4P!\4V)UDG!X05)P>X;30GQ00')3(GI2<'M`DC!X1Y%
MB<E,">E)"<A)B<Y!B<U-(=!)(=Y-"<9,BX0DJ`$``$D!P$T!X$V)S$T!\$V)
MSDG!Y`5)P>X;30GT28G.10'$38G02<'J`DG!X!Y%B>1-"=!%B<)-"<9-(<Y,
M(=%,"?%,B[0DL`$``$D!QDD!WDR)XTP!\4V)YDC!XP5)P>X;3`GS`<M,B<E)
MP>D"2,'A'HG;3`G)20G(28G.08G)32'@32'630G&3(N$)+@!``!)`<!-`>A)
MB=U-`?!)B=Y)P>4%2<'N&TT)]4F)SD4!Q4V)X$G![`))P>`>18GM30G@18G$
M30G&22'>3"'A3`GQ3(NT),`!``!)`<9-`=9-B>I,`?%-B>Y)P>(%2<'N&TT)
M\D$!RDB)V4C!ZP)(P>$>18G22`G920G(28G.B<M-(>A-(>9-"<9,BX0DR`$`
M`$D!P$T!R$V)T4T!\$V)UDG!X05)P>X;30GQ28G.10'!38GH2<'M`DG!X!Y%
MB<E-">A%B<5-"<9-(=9,(>E,"?%,B[0DT`$``$D!QDT!YDV)S$P!\4V)SDG!
MY`5)P>X;30GT00',3(G12<'J`DC!X1Y%B>1,"=%)"<A)B<Y!B<I-(<A-(>Y-
M"<9,BX0DV`$``$D!P$D!V$R)XTT!\$V)YDC!XP5)P>X;3`GS28G.1`'#38G(
M2<'I`DG!X!Z)VTT)R$T)QD6)P4TAYDPAR4P)\4R+M"3@`0``20'&30'N28G=
M3`'Q28G>2<'E!4G![AM-"?5!`<U,B>%)P>P"2,'A'D6)[4P)X4D)R$F)S$&)
MSTDAV$TAS$T)Q$R+A"3H`0``20'`30'038GJ30'@38GL2<'B!4G![!M-">)%
M`<))B=A(P>L"2<'@'D6)TDD)V$B)RT6)Q$P)PTPAZTPAX4@)V4B+G"3P`0``
M2`'#2`.$)/@!``!)`=E,B=-,`?A,B[PD&`(``$D!R4R)T4C!ZQM(P>$%2`G9
MB<E)`<E,B>E)P>T"2,'A'D2)RTP)Z4D)R$F)S4&)SDTAT$TAY4T)Q4V-1`4`
M2(G828G=2<'M&TC!X`5,">B)P$D!P$R)T$G!Z@)(P>`>18G%20G"N-;!8LI,
M,=%$,<E,BXPD``(``$D!P4T!X4V)[$P!R4V)Z4G![!M)P>$%30GA28G<18G)
M3`')2<'D'DD!QTC!ZP)!B<E,">--B=1%B=)),=Q%,<1,BX0D"`(``$D!P$T!
M\$V)SDT!X$V)S$G![AM)P>0%30GT38GN2<'M`DG!YAY%B>1-"?5,B[0D$`(`
M`$T!X$6)Q$D!QDT!\DF)WDTQ[D$QSDR)X4T!\DV)YDC!X05)P>X;3`GQB<E)
M`<I,B<E)P>D"2,'A'D6)UDD)R8G93(GK3#'+3`'Y3(N\)"`"``!$,<-,C009
M3(GQ3(GS2,'A!4C!ZQM)`<=("=F)R4D!R$R)X4G![`)(P>$>1(G#3`GA18GL
M38G-18G)23'-30'\3(N\)#@"``!%,=5/C10L28G<28G=2<'M&TG!Y`5-">Q-
MB?5)P>X"2<'E'D6)Y$T)[DR+K"0H`@``30'B18G420'%30'I28G-B<E-,?5%
M,<5-B>!-`>E-B>5)P>`%2<'M&TT)Z$6)P$T!P4F)V$C!ZP))P>`>18G-20G8
M2(N<)#`"``!(`<-(`=E,B?-,,<-$,=-)`<=,C1093(GI3(GK2,'K&TC!X05(
M"=F)R4D!RDR)X4G![`)(P>$>1(G320G,1(GQ38G&18G`33'F3`'Y3(N\)$@"
M``!%,<Y.C0PQ2(G928G>20''2<'N&TC!X05,"?&)R4D!R4R)Z4G![0)(P>$>
M18G.3`GI3(NL)$`"``!)`<5-`>A-B>5),<U%,=5/C10H38GP38GU2<'M&TG!
MX`5-">A%B<!-`<))B=A(P>L"2<'@'D6)U4P)PT6)X$F)S(G)30'X23'<3(N\
M)%@"``!%,<Q/C0P@38GH38GL20''2<'L&TG!X`5-">!%B<!-`<%-B?!)P>X"
M2<'@'D6)S$T)\$R+M"10`@``20'&3`'Q28G>33'&13'63HT4,4R)X4V)YDG!
M[AM(P>$%3`GQB<E)`<I,B>E)P>T"2,'A'D6)UDD)S8G93(G#18G`3#'K3`'Y
M3(N\)&@"``!$,<M,C0P93(GQ3(GS2,'K&TC!X05("=F)R4D!R4R)X4G![`)(
MP>$>1(G+3`GA3(ND)&`"``!)`<1-`>!-B>Q)`<=),<Q%,=1/C10@28G828G<
M2<'L&TG!X`5-">!%B<!-`<)-B?!)P>X"2<'@'D6)U$T)QD6)Z$F)S8G)33'U
M30'X3(N\)'@"``!%,<U/C0PH38G@38GE2<'M&TG!X`5-">A%B<!-`<%)B=A(
MP>L"2<'@'D6)S4D)V$B+G"1P`@``2`'#2`'93(GS3#'#1#'33(T4&4R)Z4R)
MZTC!ZQM(P>$%2`G9B<E)`<I,B>%)P>P"2,'A'D2)TTD)S$2)\4V)QDD!QTTQ
MYDP!^46)P$R+O"2(`@``13'.28G920'.2(G92<'I&TD!QTC!X05,"<F)R4D!
MSDR)Z4G![0)(P>$>18GQ3`GI3(NL)(`"``!)`<5-`>A-B>5),<U%,=5/C10H
M38G(38G-2<'M&TG!X`5-">A%B<!-`<))B=A(P>L"2<'@'D6)U4P)PT6)X$F)
MS(G)23'<30'X13'T38GN30'@38GL2<'N&TG!Y`5-"?1-B<Y%B>1-`>!)P>8>
M2<'I`D6)Q$T)\4R+M"20`@``20'&2`.$))@"``!,`?%)B=Y,`=B)VTTQSD@!
MV$4QUDV)XDP!\4V)YDG!X@5)P>X;30GR1`'138GJ2<'M`DG!XAZ)R4T)U4V)
MRD0#2B!-,>I!`?U,B4H@13'028G*3(EJ&$P!P$F)R$G!ZAL!Z4G!X`5(B4H(
M30G01`'`2(D"3(G@2<'L`DC!X!Y,">`!\$B+-;YM``!(B4(02(N$)*@"``!(
M*P9U%$B!Q+@"``!;7E]=05Q!74%>05_#Z*0%``"0#Q\`2(/L*+F`````Z$(&
M``!F#V\%^FP``$C'0#``````#Q$`9@]O!?=L``#'0'@`````#Q%`$&8/;P7T
M;```#Q%`($B#Q"C#9F8N#Q^$``````!!54%455=64TB#["A)8\!,BT$H2(G/
M2(G32(M7,$F)Q$B-=SA!C0S`3#G!2(E/*$AC3WA(@](`2,'H'4@!T$B)1S"%
MR71(O4````!(B=HIS40YY4$/3^Q(`?%,8^U-B>CHNP4``(M'>`'HB4=X@_A`
M=`Y(@\0H6UY?74%<05W#D$B)^4$I[$P!Z^ARZ/__08/\/P^.V````$&#[$!%
MB>5!P>T&08UM`4C!Y09(`=UFD/,/;P-(B?E(@\-`#Q$&\P]O2]`/$4X0\P]O
M4^`/$58@\P]O6_`/$5XPZ"+H__](.=UUS4'!Y09%*>Q)8\1(@_@(<EU(BU4`
M2(U."$B#X?A(B5<X2(M4!?A(B50&^$@ISD@!\$@I]4B#X/A(@_@(<AA(@^#X
M,=),BT05`$R)!!%(@\((2#G"<NY$B6=X2(/$*%M>7UU!7$%=PP\?@`````"H
M!'4D2(7`=-\/ME4`B%<XJ`)TU`^W5`7^9HE4!O[KR$B)W>ER____BU4`B5<X
MBU0%_(E4!OSKL`\?A```````05155U932(/L($B+:BA(BWHP2(GH2,'H`X/@
M/TB)SDR-8CA(B=.-2`%(F,9$`CB`2&/!3`'@@_DX#XZ!`0``0;A`````02G(
M=!`QTHG1@\(!Q@0(`$0YPG+R2(G9Z`[G__]F#^_`#Q%#.$G'1"0P`````$$/
M$40D$$$/$40D($B)^D`/ML5)B>A)B>E(P>H82,'@"$B)^4G!Z1!)B=)(B>I)
MP>@8#[;N10^VR44/ML!(B?I(">A(P>D02,'@"`^VR4P)R$C!X`A,"<!$#[;'
M#[;^00^VTDC!X`A,"<!(P>`(2`GX2,'@"$@)R$B)V4C!X`A("=!(B4-PZ'+F
M__](BP-(B=E(P>@8B`9(BP-(P>@0B$8!2(L#B&8"2(L#B$8#2(M#"$C!Z!B(
M1@1(BT,(2,'H$(A&!4B+0PB(9@9(BT,(B$8'2(M#$$C!Z!B(1@A(BT,02,'H
M$(A&"4B+0Q"(9@I(BT,0B$8+2(M#&$C!Z!B(1@Q(BT,82,'H$(A&#4B+0QB(
M9@Y(BT,8B$8/2(M#($C!Z!B(1A!(BT,@2,'H$(A&$4B+0R"(9A)(BT,@B$83
M2(/$(%M>7UU!7.E1`@``D+HX````13')*<J#^@AS*?;"!'5LA=(/A*/^___&
M``#VP@(/A)?^__\QR6:)3!#^Z8O^__\/'T``B=%(QP``````2,=$"/@`````
M2(U("$B#X?A(*<@!PH/B^(/Z"`^"7/[__X/B^#'`08G`@\`(3HD,`3G0<O+I
M1/[__P\?1```QP``````QT00_`````#I+/[__Y"0D)"0D)"0D)"0D)#_)4+-
M``"0D/\E,LT``)"0_R4BS0``D)#_)1+-``"0D/\E`LT``)"0_R7RS```D)#_
M)>+,``"0D/\ERLP``)"0_R6ZS```D)#_):K,``"0D/\EFLP``)"0_R6*S```
MD)#_)7K,``"0D/\E:LP``)"0_R5:S```D)#_)4K,``"0D/\E.LP``)"0_R4J
MS```D)#_)1K,``"0D/\E"LP``)"0_R5RR@``D)"0D)"0D)"0D/\E4LL``)"0
MD)"0D)"0D)#_)<K+``"0D`\?A```````45!(/0`0``!(C4PD&'(92('I`!``
M`$B#"0!(+0`0``!(/0`0``!WYT@IP4B#"0!86<.0D)"0D)"0D)"0D)"0D/\E
M\LD``)"0D)"0D)"0D)#_)>K)``"0D)"0D)"0D)"0_R7JR0``D)"0D)"0D)"0
MD/\EXLD``)"0D)"0D)"0D)#_)=K)``"0D)"0D)"0D)"0_R72R0``D)"0D)"0
MD)"0D/\EXLD``)"0D)"0D)"0D)#_)>+)``"0D)"0D)"0D)"0_R7ZR0``D)"0
MD)"0D)"0D/\E\LD``)"0D)"0D)"0D)#_)>K)``"0D)"0D)"0D)"0_R7BR0``
MD)"0D)"0D)"0D/\EVLD``)"0D)"0D)"0D)#_)=+)``"0D)"0D)"0D)"0_R7*
MR0``D)"0D)"0D)"0D/\EPLD``)"0D)"0D)"0D)#_);K)``"0D)"0D)"0D)"0
M_R6RR0``D)"0D)"0D)"0D/\EJLD``)"0D)"0D)"0D)#_):+)``"0D)"0D)"0
MD)"0_R6:R0``D)"0D)"0D)"0D/\EDLD``)"0D)"0D)"0D)#_)9K)``"0D)"0
MD)"0D)"0_R62R0``D)"0D)"0D)"0D/\EBLD``)"0D)"0D)"0D)#_)8+)``"0
MD)"0D)"0D)"0_R5ZR0``D)"0D)"0D)"0D/\E<LD``)"0D)"0D)"0D)#_)6K)
M``"0D)"0D)"0D)"0_R5BR0``D)"0D)"0D)"0D/\E6LD``)"0D)"0D)"0D)#_
M)5+)``"0D)"0D)"0D)"0_R5*R0``D)"0D)"0D)"0D/\E0LD``)"0D)"0D)"0
MD)#_)3K)``"0D)"0D)"0D)"0_R4RR0``D)"0D)"0D)"0D/\E*LD``)"0D)"0
MD)"0D)!(@^PH,=+H50```/\5U\<``)"0D)"0D)"0D)"0D)"0D,.0D)"0D)"0
MD)"0D)"0D)##D)"0D)"0D)"0D)"0D)"0PY"0D)"0D)"0D)"0D)"0D,.0D)"0
MD)"0D)"0D)"0D)!64TB#["A(B<Y(B=.Y"````.C*!```2(7;#X0T`@``2+H`
M````90$``$B)D_````!(C0TY!P``2(E+2$B-%48'``!(B5-02(L50\<``$B)
M4QA(B7-`2(TU9/___TB)LX````!(C35F____2(FSB````$B-#6C___](B8N0
M````2(TU:O___TB)LY@```!(C14,I```2(E3.$C'`P````!(N4@!``"^"P``
M2(E+",=#$`,```!(C365_?__2(ES($B-%2K]__](B5,H2(T-#_[__TB)2S!(
MC37D`P``2(ES>$B-%0D$``!(B9,8`0``2(7`#X3E````2(N`X````$B%P`^$
MU0```(L5HZ,``(72#X7'````2(,]HQ,````/A"8"``!(@SV=$P````^$AP(`
M`$B#/9<3````#X1I`@``2(,]D1,````/A$L"``!(@SV+$P````^$+0(``$B#
M/843````#X0/`@``2(,]?Q,````/A/$!``!(@SUY$P````^$TP$``$B+%303
M``!(B1!(BQ4R$P``2(E0"$B+%2\3``!(B5`02(L5+!,``$B)4!A(BQ4I$P``
M2(E0($B+%283``!(B5`H2(L5(Q,``$B)4#!(BQ4@$P``2(E0.#')Z%T#``!(
MB8/H````2(T%CQ(``$B)0UA(C04$$P``2(E#8$B-!7FB``!(B4-H2(T%3J0`
M`$B)0W!(C06C90``2(F#(`$``$B-!=%E``!(B8,H`0``2(T%1[+__TB)@S`!
M```QR>AI`@``N`$```!(@\0H6U[#2+D`````90$``$B)B/````!(C0T%!0``
M2(E(2$B-%1(%``!(B5!02(N(0`$``$B+%0C%``!(B0I(B7!`2(TU*OW__TB)
ML(````!(C34L_?__2(FPB````$B--2[]__](B;"0````2(TU,/W__TB)L)@`
M``!(C372H0``2(EP.$C'``````!(OD@!``"^"P``2(EP",=`$`,```!(C35;
M^___2(EP($B-#?#Z__](B4@H2(TUU?O__TB)<#!(C16J`0``2(E0>$B-#<\!
M``!(B8@8`0``2(7`#X2K_O__2(G#2(N`X````$B%P`^%P_W__^F3_O__2(L0
M2(D5;1$``.G+_?__2(M0.$B)%941``#I'?[__TB+4#!(B15]$0``Z?_]__](
MBU`H2(D591$``.GA_?__2(M0($B)%4T1``#IP_W__TB+4!A(B14U$0``Z:7]
M__](BU`02(D5'1$``.F'_?__2(M0"$B)%041``#I:?W__Y"0D)"0D)"03(L%
MZ:```(L5ZZ```$B+#>R@``#IMP```$B#[#B#^@)T:W=,A=)T<TB)#=&@``#'
M!;^@```!````3(D%L*```#'`387`#Y3`B062H```2(T5J____^C6````2(D%
MAZ```$B#^/\/E<`/ML!(@\0XPX/Z`W4.N@,```!(@\0XZ4T```"X`0```$B#
MQ#C#N@(```!(@\0XZ34````QTN@N````A<!TQ8E$)"Q(BPTWH```Z$H```!(
MQP4GH```_____XM$)"SKI)"0D)"0D)"0D+@!````PY"0D)"0D)"0D)##D)"0
MD)"0D)"0D)"0D)"0_R7:P@``D)"0D)"0D)"0D/\EVL(``)"0D)"0D)"0D)#_
M)=+"``"0D)"0D)"0D)"0_R5*PP``D)"0D)"0D)"0D%932(/L*$B)RTB)T4B-
M-<V?``!(B?+HY?K__TB)\DB)V4B#Q"A;7ND$````D)"0D/\EBL(``)"0D)"0
MD)"0D)#_)5+$``"0D`\?A```````5U932(/L4(E,)'!(B50D.$R)1"0PZ`;X
M__](C50D.$R-1"0P2(L=%6```$B-3"1P2(L#2(E$)$@QP.CQ]O__2(L5VE\`
M`$B+#<-?``!,C04L]___Z$?W__](BP6@7P``2(LUR5\``(`X`'4VZ*?V__](
MB070G@``2(G!2(7`#X1W`0``Z(?V__](BS6@7P``2(GQZ!CW__](BP#&@"X)
M````2(GQZ`;W__](BP"`B'($```#BT0D<(U(!(UP`TACR4C!X0/H?O;__TR+
M5"0X1(M$)'!)B<%(B05BG@``28L"28D!2(T%-B$``$F)00A(BP5:#@``28E!
M$$B-!2,A``!)B4$808/X`0^.X@```$F#P@A!@\`#N@0````/'T``28L"2&/*
M@\(!28/""$F)!,E$.<)UZDACTDB+#0B>``!!B?!)QP31`````$B-%4[$__](
MQT0D(`````#HF/7__X7`=$Q(BPW=G0``2(U\)$3HD_7__TB)^DB-#4P>``")
MQN@JOO__2(7`=#-(BPVVG0``Z&GU___HC/7__TB+1"1(2"L#=5^)\$B#Q%!;
M7E_#2(L-D9T``.@T]?__ZZ9(BPUSG0``2(7)=`?H8=G__^NZ2(T-H1T``$B)
M^NC0O?__2(D%49T``$B)P4B%P'2<Z]FZ!````.D[____N0$```#H>_;__^A&
M]O__D)"0D)"0Z6N]__^0D)"0D)"0D)"0D/__________X%)```$`````````
M`````/__________````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````````I0
M05(N<&T*```````````(<T```0``````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````````E
M<R\E<P`O<')O8R]S96QF+V5X90!$>6YA3&]A9&5R.CIB;V]T7T1Y;F%,;V%D
M97(``$EN=&5R;F%L<SHZ4$%2.CI"3T]4`"X`4$%27U1%35``4$%27U!23T=.
M04U%`#H`)7,E<R5S`"\`4$523#5,24(`4$523$Q)0@!015),-4]05`!015),
M24\`4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$`%!!4E]#3$5!3@!005)?
M1$5"54<`4$%27T-!0TA%`%!!4E]'3$]"04Q?1$5"54<`4$%27T=,3T)!3%]4
M35!$25(`4$%27U1-4$1)4@!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#
M3$5!3@`Q`&QS965K(&9A:6QE9`!R96%D(&9A:6QE9`!0051(`%-94U1%30`O
M=&UP`%1-4$1)4@!414U01$E2`%1%35``5$U0`%5315(`55-%4DY!344`)3`R
M>``E<R5S)7,E<P!P87(M```````````E<SH@8W)E871I;VX@;V8@<')I=F%T
M92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*````````)7,Z
M('!R:79A=&4@<W5B9&ER96-T;W)Y("5S(&ES('5N<V%F92`H<&QE87-E(')E
M;6]V92!I="!A;F0@<F5T<GD@>6]U<B!O<&5R871I;VXI"@!P87)L+F5X90`N
M<&%R`'-H;W)T(')E860``$-!0TA%`"5S)7-C86-H92TE<R5S`"5S)7-T96UP
M+25U)7,`)7,E<W1E;7`M)74M)74E<P`8`'!E<FP`,`!005(Z.E!A8VME<CHZ
M5D524TE/3@`Q+C`V-`````!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C
M=&]R>0``````````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B
M9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@`M90`M+0``<&5R;'AS
M:2YC`````````&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ
M*R(D0")])PII9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*
M<&%C:V%G92!?7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E
M=%]S:7IE+"`D8V%C:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#
M(#T@(EQN4$%2+G!M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P
M86-K*").(BD*)&-A8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@
M/2`B4$M<,#`S7#`P-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F
M:"D@/2!`7SL*;7D@)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68[
M"FUY("1S:7IE(#T@+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S
M:7IE+3$I("4@)&-H=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@
M;75L=&EP;&4@;V8@)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS
M965K("1F:"P@)'!O<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE
M("L@;&5N9W1H*"1005)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B
M=68L("1005)?34%'24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*
M)'!O<R`M/2`D8VAU;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?
M=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$
M('L@:68@*"1%3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`[
M"G)E<75I<F4@1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*
M;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M
M<"D["F]U=',H<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*
M1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@
M=6YL:6YK('TL("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD
M("1P87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N
M97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^
M($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@
M)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^
M9FEL96YA;64["G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD
M>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@
M)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@
M7"1X("L@,6`*9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*
M8VAM;V0@,#<P,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^
M+V1E=B]N=6QL(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W
M;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F
M:6QE;F%M95TI.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4
M*"D@:68@9&5F:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?
M<&%R7VEN:71?96YV*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT[
M"FUY("5S>7,@/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@
M/R`G.R<@.B`G.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7
M:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]
M/B`H)%Y/(#U^("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T
M7W!R;V=N86UE*"D["F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B
M72D["E]S971?<&%R7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@
M/2`B)$5.5GM005)?5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?
M;V9?1DE,15]S96-T:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?
M7WT@/2!S=6(@>WT["G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G
M;F%M92D@>PIO=71S*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA
M;64B72D[("`C(&1O;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R
M<FYO+G!M"FQA<W0@34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R
M7VUA9VEC*"I?1D@I.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#
M86XG="!F:6YD(&UA9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B
M)'!R;V=N86UE(ETI.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M
M86=I8R!A="!P;W-I=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G
M:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B
M*2P@,#L*<F5A9"!?1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H
M)&)U9B!N92`B7#!#04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T
M<R@B3F\@8V%C:&4@;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T
M<RAQ<5M#86-H92!M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D
M;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D
M8G5F+"`D1DE,15]O9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B
M3B(L("1B=68I.PIO=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I
M<R`D;V9F<V5T(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F
M<V5T7W-I>F4@+2`D;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N
M(#T@=&5L;"!?1D@["FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F
M+"`T.R`@("`@("`@("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER
M<W0@(D9)3$4B"G=H:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@
M)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM
M>2`D9G5L;&YA;64@/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D
M9G5L;&YA;64B+BXN72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH
M6V$M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L
M("1E>'0I(#T@*"1B=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A
M9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D
M8G5F*3L*:68@*&1E9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\
M<&Q\:7A\86PI)"]I*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C
M)&5X="(L("1B=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F
M=6QL;F%M97T@/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[
M)&9I;&5N86UE?2`]("1F=6QL;F%M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE
M>R1F=6QL;F%M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M
M92`]/B`D9G5L;&YA;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK
M.PI]"G)E860@7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*
M;7D@*"1S96QF+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U
M;&4@;W(@(21M;V1U;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U
M;&5].PIR971U<FX@=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES
M='LD;6]D=6QE?3L*)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M
M;V1U;&4B.PII9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ
M1FEL93HZ;F5W*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@
M;W(@9&EE(")#86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN
M;6]D92@I.PHD9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P
M+"`P*3L*<F5T=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A
M=F5?87,H(B1I;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O
M+3Y[9FEL97T@/2`D9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D
M9FEL96YA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A
M73L*<F5T=7)N("1F:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C
M86XG="!F:6YD(&UO9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E
M(%A33&]A9&5R.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ
M2&5A=GD["G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?
M:6YI=%]D>6YA;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM
M>2`D9FEL96YA;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2![
M"G5N;&5S<R`H)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$
M4$%.+RD@>PII9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER
M92`D9FEL96YA;64["GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I
M;&5N86UE(#U^("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R
M97%U:7)E7VQI<W1[)&9I;&5N86UE?3L*?0II9B`H)&)U9B!N92`D4$M:25!?
M34%'24,I('L*;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T
M:6]N(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T
M87)T7W!O<R`]("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T
M87)T(&]F('II<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L
M("1L;V=F:"P@)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%
M55-%?3L@(R!S86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O
M8FQE;2D*:68@*"$D<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M
M;W!T:6]N<R<@)B8@<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@
M8FQI8E]T;U]P87(*:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R
M"G,@("!S:6=N7W!A<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD
M("1!4D=66S!=(&5Q("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM0
M05)?05!07U)%55-%?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA
M;"!P87)L(&)E:&%V:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6
M*2!["B1!4D=66S!=(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R
M(&QA<W0["FEF("@D,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#([
M"GT*96QS:68@*"0Q(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF
M("@D,2!E<2`G02<I('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF
M("@D,2!E<2`G3R<I('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I
M('L*)&)U;F1L92`]("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B
M=6YD;&4@/2`G86QL)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`]
M(#$["GT*96QS:68@*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@
M)#(@;W(@9&EE('%Q6T-A;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]
M"F5L<VEF("@D,2!E<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF
M="A`05)'5BD["FEF("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE
M=&4@)$5.5GLG4$%27U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z
M.D1I<W0["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF
M>R)005(Z.D1I<W0Z.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*
M=6YS:&EF="!`24Y#+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*
M<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I
M<F4@1&EG97-T.CI32$$["GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@
M)'II<#L*:68@*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W
M)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z
M("0A73L*8FQE<W,H)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z
M.EII<"T^;F5W.PHH("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P
M87(I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O
M<B!R96%D:6YG('II<"!A<F-H:79E("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R
M(#T@9&\@>PII9B`H)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT
M<R@G345402YY;6PG*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE
M=&$@/7X@<R]>7%,N*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI
M)"]M9SL*?0I].PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G
M/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D
M<&%R(CH@)"%=.PIM>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q
M6R(D<&%R(B!I<R!N;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@)%!+
M6DE07TU!1TE#.PIC;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY
M("1F:"`]($E/.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4
M*"D@?"!)3SHZ1FEL93HZ3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I
M+`HP-S<W+`HI(&]R(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z
M("0A73L*)&9H+3YB:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A
M9&5R.PII9B`H9&5F:6YE9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*2!["G)E
M860@7T9(+"`D;&]A9&5R+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N.PI](&5L
M<V4@>PIL;V-A;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@
M*"$D14Y6>U!!4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<
M0')E;2DO*2!["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z
M.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H
M+3YP<FEN="@D;&]A9&5R*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D
M97(@=&\@(B1O=70B.B`D(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2
M.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET
M7VEN8R@I.PIR97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A
M+T)31%!!3B\@?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@
M;F4@)$-O;F9I9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N
M9FEG.CI#;VYF:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S
M?"\D?'P@9F]R96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F
M;W(@0$1Y;F%,;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.
M0SL*;7D@)&QI8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@
M("`@("`@(R!86%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF
M;W)E86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE
M+"`D;F%M92D["FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&
M=6QL0V%C:&5[)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F
M8RD["GT*96QS92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME
M>2`]?B!M?%Y<421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@
M*"1K97DL("0Q*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>
M+UTK+R@N*BDD?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI
M('L**"1F:6QE+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M
M9B`B)&1I<B\D,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D
M,2D["FQA<W0["GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*
M;F5X="!I9B`D=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE
M*2!A;F0@)&9I;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN
M=&8H<7%;4&%C:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL
M92T^>VYA;65](#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I
M;&4I*2!["B1C;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A
M;"`D+R`]('5N9&5F.PIO<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@
M9&EE('%Q6T-A;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD
M=&@^.PIC;&]S92`D=&@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH
M7"1C;VYT96YT+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]6
M15)"051)37T@86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2
M.CI&:6QT97(Z.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L
M("1F:6QE+"`D;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG
M+"!L96YG=&@H)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H
M:79E.CI::7`Z.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C
M:R@G3B<L(&QE;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q
M6T5R<F]R('=R:71I;F<@96UB961D960@1DE,12!T;R`B)&]U="(Z("0A73L*
M;W5T<RAQ<5M7<FET=&5N(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2![
M"B1Z:7`M/G=R:71E5&]&:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP
M.CI!6E]/2PIO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B
M)&]U=")=.PI]"FEF("@D;65T85]P87)[8VQE86Y]*2!["B1F:"T^<')I;G0H
M(EPP0TQ%04XB*3L*?0IE;'-E('L*:68@*"%D969I;F5D("1C86-H95]N86UE
M*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K("1F:"P@
M,"P@,#L*)&-T>"T^861D9FEL92@D9F@I.PIS965K("1F:"P@,"P@,CL*)&-A
M8VAE7VYA;64@/2`D8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S
M=6)S='(@)&-A8VAE7VYA;64L(#`L("1C86-H95]N86UE7W-I>F4["B1C86-H
M95]N86UE("X](")<,"(@>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D
M8V%C:&5?;F%M92D["B1F:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%
M(BD["GT*;7D@)&]F9G-E="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R
M*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I
M.PHD9F@M/F-L;W-E(&]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL97(@
M;V8@(B1O=70B.B`D(5T["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL
M87-T('5N;&5S<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H
M*3L*<F5Q=6ER92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H
M*3L*>PIR97%U:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP
M.PI]"FUY("1F:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@
M("`@("`@("`@("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ
M2&%N9&QE"B1F:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q
M<5MF9&]P96XH*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N
M:U-I>F4H+7,@7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH
M)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!
M<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG
M('II<"!A<F-H:79E("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#
M:'5N:U-I>F4H-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II
M<#L*)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H(D5X
M=')A8W1I;F<@>FEP+BXN(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]
M*2![(",@<VAO=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY
M("1M96UB97(@*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M
M/FES1&ER96-T;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE
M3F%M93L*;F5X="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI
M8B\**#\Z)$-O;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*
M?7@["FUY("1E>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I
M;&4Z.E-P96,M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA
M;64I.PII9B`H+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N
M8V]M<')E<W-E9%-I>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R
M7VYA;64B('-I;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE
M(ETI.PI](&5L<V4@>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M
M92(@=&\@(B1D97-T7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&
M:6QE3F%M960H)&1E<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I
M*0IO<B!D:64@<7%;17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D
M9&5S=%]N86UE(ET["F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E
M<2`B:'!U>"(["GT*?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O
M9VYA;65]*2!["F1I92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@
M6R`M06QI8BYP87(@72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A
M<B!=(%L@<')O9W)A;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S
M<F,N<&%R"BX*)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@
M/2!S:&EF="A`05)'5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I
M(#T@0%\["G)E<75I<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L
M("1P871H+"`D97AT*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA
M;64L("@G7"XN*B<I*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H
M.CIM:W!A=&@H)'!A=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I
M97,@=VET:"!E<G)O<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E
M(&QI8CL*<F5Q=6ER92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E
M<75I<F4@<W1R:6-T.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,[
M"G)E<75I<F4@0V%R<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%
M<G)N;SL*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T
M97(["G)E<75I<F4@1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER
M92!&:6QE.CI3<&5C.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I
M9SL*<F5Q=6ER92!)3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U
M:7)E($-O;7!R97-S.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q
M=6ER92!$:6=E<W0Z.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(
M96%V>3L*<F5Q=6ER92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z
M.E!O9%-T<FEP.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0[
M"G)E<75I<F4@871T<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V
M86P@>R!R97%U:7)E(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ
M571I;"!].PIE=F%L('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@
M?3L*979A;"![(')E<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*
M979A;"![(')E<75I<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R
M('T["F5V86P@>R!R97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M
M<"!["FEF("AD969I;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]4
M14U0?2`]?B`O*"XK*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF
M;W)E86-H(&UY("1P871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$
M25(@5$U01$E2(%1%35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O
M=&UP("X@*0HI('L*;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@
M)'!A=&@@86YD("UW("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*
M979A;"![*"1P=W5I9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM]
M.PII9B`H(&1E9FEN960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA
M;64@/2`F5VEN,S(Z.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U
M:60I('L*)'5S97)N86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@
M/2`D14Y6>U5315).04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]
M"FUY("1S=&UP9&ER(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K
M*")(*B(L("1U<V5R;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D
M8V%C:&5?9&ER.PII9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@
M/2`B=&5M<"TD)"(["GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D
M<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A
M73L*:68@*"AM>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@
M/CT@,"D@>PIS965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S
M:7IE"BT@;&5N9W1H*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H
M+"`D8G5F+"!L96YG=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#
M2$4B*2!["G-E96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I
M>F4*+2!L96YG=&@H(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`[
M"G)E860@)&9H+"`D8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O
M7#`O+V<["B1C86-H95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D
M9F@["GT*:68@*"$D8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD
M)"(["B1%3E9[4$%27T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[
M7V1E;&EM?21C86-H95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.
M5GM005)?5$5-4'T@/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D
M,2!I9B`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N
M*RDO.PI]"G-U8B!?<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D
M;6]D92D@/2!`7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92([
M"G5N;&5S<R`H+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L
M;&YA;64N)"0B.PIM>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N
M86UE*0HF)B`H<')I;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*
M;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM
M;V0@)&UO9&4L("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H
M)'1E;7!N86UE+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]
M"G)E='5R;B`D9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H
M9&5F:6YE9"`D14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.
M04U%?2`]?B`O*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE
M('Q\/2`D,#L*:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA
M;64L("1%3E9[4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T
M<B@D<')O9VYA;64L(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@
M*R`Q*3L*?0II9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O
M9VYA;64L("1S>7-[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L
M("<\)RP@)'!R;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US
M("(D<')O9VYA;64D<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?
M97AE?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D
M<WES>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T
M<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*
M)&1I<B`]?B!S+UQ1)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@
M(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I
M"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2([
M"B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@
M;&%S="D*:68@+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*
M<W5B(%]F:7A?<')O9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM0
M05)?4%)/1TY!345].PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI
M;7TI(#P@,"D@>PHD<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M
M92(["GT*;7D@)'!W9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z
M.F=E=&-W9"@I"CH@*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R
M.CI'971#=V0H*2`Z(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]
M?B!S+UXH/SU<+EPN/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI
M;7TO.PHD14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P
M87)?:6YI=%]E;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T]
M(#$@*2!["G)E='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141]
M(#T@,CL*?0IF;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#
M2$4@4%)/1TY!344@*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O
M<B`H<7<O(%1-4$1)4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%2
M7R1?(GT@/2`D14Y6>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[
M(E!!4E]'3$]"04Q?)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L
M971E("1%3E9[4$%27T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF
M("1Q=6EE=#L*:68@*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]
M"F5L<V4@>PIP<FEN="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U
M:7)E($-O;F9I9SL*<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N
M9FEG.CI#;VYF:6=[)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I
M=&5A<F-H97AP('-I=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E
M>'`**3L*?0IP86-K86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)
M3D,L(%PF4$%2.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*
M9&EE('%Q6W!A<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N
M86UE(CH@3F\@<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@
M)'!R;V=N86UE.PID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D
M0"`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*
M)#HZ7U]%4E)/4B`]("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D
M.CI?7T524D]2(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?
M15)23U(@:68@)#HZ7U]%4E)/4CL*,3L*7U]%3D1?7PH```````````#_____
M`````/____\``````2-%9P````")J\WO`````/[<NI@`````=E0R$`````#P
MX=+#````````````````,%```````````````````&`400`!````````````
M``!82$```0``````````````4$A```$``````````````-`400`!````````
M````````$T$``0``````````````\$=```$``````````````$@300`!````
M``````````!'0T,Z("A'3E4I(#$R+C0N,````````````````````$=#0SH@
M*$=.52D@,3(N-"XP````````````````````1T-#.B`H1TY5*2`Q,BXT+C``
M``````````````````!'0T,Z("A'3E4I(#$R+C0N,```````````````````
M`$=#0SH@*$=.52D@,3(N-"XP````````````````````1T-#.B`H1TY5*2`Q
M,BXT+C````````````````````!'0T,Z("A'3E4I(#$R+C0N,```````````
M`````````$=#0SH@*$=.52D@,3(N-"XP````````````````````1T-#.B`H
M1TY5*2`Q,BXT+C````````````````````!'0T,Z("A'3E4I(#$R+C0N,```
M`````````````````$=#0SH@*$=.52D@,3(N-"XP````````````````````
M1T-#.B`H1TY5*2`Q,BXT+C````````````````````!'0T,Z("A'3E4I(#$R
M+C0N,````````````````````$=#0SH@*$=.52D@,3(N-"XP````````````
M````````1T-#.B`H1TY5*2`Q,BXT+C````````````````````!'0T,Z("A'
M3E4I(#$R+C0N,````````````````````````````````0```&`4`0#0L```
M0````-`4`0``L0``0``````3`0`0L0``0````$@3`0`PL0``0```````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````(````9````',```!R.``!24T13\P,_0CW;F<F"?>:]/Y7QQP$`````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````!```$00````X```4!```&40```(X```<!```'$0```,X```
M@!```%(1```0X```8!$``-\1```DX```X!$``$T3```PX```4!,``"T4``!$
MX```,!0``&,6``!8X```<!8``-(6``!PX```X!8``!$7``!XX```(!<``",7
M``"`X```,!<``$09``"$X```4!D``'H9``"<X```@!D``*H:``"DX```L!H`
M`*$?``"TX```L!\``!(@``#,X```("```'(@``#4X```@"```)LA``#@X```
MH"$``+TA``#PX```P"$``(LB``#XX```D"(``-(I```,X0``X"D``/,K```D
MX0```"P``,,L```TX0``T"P``'U#``!,X0``@$,``,5#``!DX0``T$,``%A%
M``!LX0``8$4``+-'``"`X0``($L``#)+``"0X0``0$L``$%+``"8X0``4$L`
M`%%+``"<X0``8$L``&%+``"@X0``<$L``'%+``"DX0``@$L``$A/``"HX0``
M4$\``&E/``"TX0``:4\``!=0``"XX0``(%```"90``#`X0``,%```#%0``#$
MX0``@%```*Q0``#(X0``T%```-M2``!`X0``X%(``.52``#4X0``````````
M```````````````````````````````````!!`$`!$(```$````!`````0P'
M``Q""#`'8`9P!5`$P`+0```!"00`"0$7``(P`6`!#P@`#P$9``@P!V`&<`50
M!,`"T`$3!P`3`08""S`*8`EP"%`"P````1`)`!""##`+8`IP"5`(P`;0!.`"
M\````04"``52`3`!!`$`!$(```$````!&0H`&0$;`A$P$&`/<`Y0#<`+T`G@
M`O`!!0(`!3(!,`$1!@`1`0<""3`(8`=P`5`!$`D`$&(,,`M@"G`)4`C`!M`$
MX`+P```!!0(`!5(!,`$&`P`&8@(P`6````$1!@`1`0<@"3`(8`=P`5`!!`$`
M!$(```$,!P`,H@@P!V`&<`50!,`"T````1,*`!,!-0`,,`M@"G`)4`C`!M`$
MX`+P`0@%``AB!#`#8`)P`5````$'!``'4@,P`F`!<`$'!``'D@,P`F`!<`$3
M"@`3`5<`##`+8`IP"5`(P`;0!.`"\`$$`0`$0@```0P'``Q""#`'8`9P!5`$
MP`+0```!"@8`"C(&,`5@!'`#4`+``00!``1"```!`````0````$````!````
M`08#``9"`C`!8````0````$$`0`$8@```0````$````!!@,`!D(",`%@```!
M````````````````````````````````````````````````````````````
M``#%^7-H`````-P``0`!````$@```!(````H``$`<``!`+@``0#@*0``H"$`
M`-!0```@8```4!D````L```@%P``("```(`9``"P'P``,!<``+`:``"0(@``
MP"$``&!%``"`0P``T$,``.`6``#D``$`^@`!`!4!`0`:`0$`)`$!`#$!`0`]
M`0$`3@$!`%8!`0!B`0$`<`$!`'T!`0"*`0$`EP$!`*D!`0"S`0$`O`$!`,<!
M`0````$``@`#``0`!0`&``<`"``)``H`"P`,``T`#@`/`!``$0!P87(N97AE
M`%A37TEN=&5R;F%L<U]005)?0D]/5`!G971?=7-E<FYA;65?9G)O;5]G971P
M=W5I9`!M86EN`&UY7W!A<E]P;`!P87)?8F%S96YA;64`<&%R7V-L96%N=7``
M<&%R7V-U<G)E;G1?97AE8P!P87)?9&EE`'!A<E]D:7)N86UE`'!A<E]E;G9?
M8VQE86X`<&%R7V9I;F1P<F]G`'!A<E]I;FET7V5N=@!P87)?;6MT;7!D:7(`
M<&%R7W-E='5P7VQI8G!A=&@`<VAA7V9I;F%L`'-H85]I;FET`'-H85]U<&1A
M=&4`>'-?:6YI=```````````````````````````````````````````````
M````````````````````:!`!````````````%!H!`,@2`0#P$0$`````````
M```D&@$`4!0!```2`0```````````)`:`0!@%`$`N!(!````````````I!H!
M`!@5`0`````````````````````````````````H%0$``````#@5`0``````
M1!4!``````!4%0$``````&`5`0``````<!4!``````"`%0$``````)05`0``
M````J!4!``````"T%0$``````,05`0``````T!4!``````#<%0$``````.@5
M`0``````_!4!```````0%@$``````"`6`0``````+!8!```````T%@$`````
M`$`6`0``````2!8!``````!4%@$``````&`6`0``````;!8!``````!T%@$`
M`````'P6`0``````B!8!``````"4%@$``````*`6`0``````J!8!``````"P
M%@$``````+P6`0``````T!8!``````#D%@$``````.P6`0``````^!8!````
M```$%P$``````!`7`0``````&!<!```````@%P$``````"P7`0``````.!<!
M``````!$%P$``````%`7`0``````7!<!``````!H%P$``````'07`0``````
M@!<!`````````````````(P7`0````````````````"D%P$``````+07`0``
M````R!<!``````#@%P$``````/@7`0``````"!@!```````<&`$``````"P8
M`0``````/!@!``````!0&`$``````&@8`0``````>!@!``````",&`$`````
M`*`8`0``````L!@!``````#0&`$``````.08`0``````]!@!```````(&0$`
M`````!@9`0``````)!D!```````T&0$`````````````````0!D!````````
M`````````"@5`0``````.!4!``````!$%0$``````%05`0``````8!4!````
M``!P%0$``````(`5`0``````E!4!``````"H%0$``````+05`0``````Q!4!
M``````#0%0$``````-P5`0``````Z!4!``````#\%0$``````!`6`0``````
M(!8!```````L%@$``````#06`0``````0!8!``````!(%@$``````%06`0``
M````8!8!``````!L%@$``````'06`0``````?!8!``````"(%@$``````)06
M`0``````H!8!``````"H%@$``````+`6`0``````O!8!``````#0%@$`````
M`.06`0``````[!8!``````#X%@$```````07`0``````$!<!```````8%P$`
M`````"`7`0``````+!<!```````X%P$``````$07`0``````4!<!``````!<
M%P$``````&@7`0``````=!<!``````"`%P$`````````````````C!<!````
M`````````````*07`0``````M!<!``````#(%P$``````.`7`0``````^!<!
M```````(&`$``````!P8`0``````+!@!```````\&`$``````%`8`0``````
M:!@!``````!X&`$``````(P8`0``````H!@!``````"P&`$``````-`8`0``
M````Y!@!``````#T&`$```````@9`0``````&!D!```````D&0$``````#09
M`0````````````````!`&0$`````````````````$`!?7V-X85]A=&5X:70`
M`!<`7U]E<G)N;P```"0`7U]G971R965N=``````O`%]?;6%I;@`````R`%]?
M;65M;6]V95]C:&L`30!?7W-P<FEN=&9?8VAK`%``7U]S=&%C:U]C:&M?9F%I
M;```4@!?7W-T86-K7V-H:U]G=6%R9`!M`%]D;&Q?8W)T,`!T`%]I;7!U<F5?
M<'1R````A`!A8V-E<W,`````!0%C86QL;V,`````1P%C;&]S961I<@``?P%C
M>6=W:6Y?9&5T86-H7V1L;`"!`6-Y9W=I;E]I;G1E<FYA;````)T!9&QL7V1L
M;&-R=#````"Z`65N=FER;VX```#7`65X:70``$,"9G!R:6YT9@```$\"9G)E
M90``M@)G971P:60`````P0)G971P=W5I9```SP)G971U:60`````DP-L<V5E
M:P"5`VQS=&%T`)H#;6%L;&]C`````*L#;65M8VAR`````*T#;65M8W!Y````
M`+,#;6MD:7(`]0-O<&5N``#Y`V]P96YD:7(````.!'!O<VEX7VUE;6%L:6=N
M`````#<$<'1H<F5A9%]A=&9O<FL`````Q01R96%D``#&!')E861D:7(```#(
M!')E861L:6YK``#+!')E86QL;V,```#Y!')M9&ER`)<%<W1A=```H@5S=')C
M:'(`````I`5S=')C;7``````J05S=')D=7``````LP5S=')L96X`````N`5S
M=')N8VUP````P`5S=')R8VAR````R@5S=')T;VL`````/P9U;FQI;FL`````
M509V9G!R:6YT9@``+0!?7V5M=71L<U]G971?861D<F5S<P``&`!03%]D;U]U
M;F1U;7```&\!4&5R;%]A=&9O<FM?;&]C:P``<`%097)L7V%T9F]R:U]U;FQO
M8VL`````_`%097)L7V-R;V%K7VYO8V]N=&5X=```D`)097)L7V=E=%]S=@``
M`+T"4&5R;%]G=E]F971C:'!V````CP-097)L7VUG7W-E=````!$$4&5R;%]N
M97=84P`````3!%!E<FQ?;F5W6%-?9FQA9W,``/\%4&5R;%]S869E<WES;6%L
M;&]C`````.(&4&5R;%]S=E]S971P=@#I!E!E<FQ?<W9?<V5T<'9N`````!T'
M4&5R;%]S>7-?:6YI=#,`````'@=097)L7W-Y<U]T97)M`*D'7U]E;75T;'-?
M=BY03%]C=7)R96YT7V-O;G1E>'0`K`=B;V]T7T1Y;F%,;V%D97(```"[!W!E
M<FQ?86QL;V,`````O0=P97)L7V-O;G-T<G5C=`````"^!W!E<FQ?9&5S=')U
M8W0`OP=P97)L7V9R964`P`=P97)L7W!A<G-E`````,$'<&5R;%]R=6X``)X"
M1V5T36]D=6QE2&%N9&QE00```!`!```0`0``$`$``!`!```0`0``$`$``!`!
M```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$`
M`!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``
M$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$``!`!```0
M`0``$`$``!`!```0`0``$`$``!`!```0`0``$`$`8WEG=VEN,2YD;&P`%!`!
M`&-Y9V=C8U]S+7-E:"TQ+F1L;```*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`
M*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0`H$`$`*!`!`"@0`0`H
M$`$`*!`!`"@0`0`H$`$`*!`!`&-Y9W!E<FPU7S0P+F1L;``\$`$`2T523D5,
M,S(N9&QL````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````$``,`
M```P``"`#@```,```(`0````\```@!@````@`0"````````````````````$
M``$```!@``"``@```'@``(`#````D```@`0```"H``"`````````````````
M```!``D$``!0`0`````````````````````!``D$``!@`0``````````````
M```````!``D$``!P`0`````````````````````!``D$``"``0``````````
M`````````0```,`!`(#8``"````````````````````!``D$``"0`0``````
M```````````````!``$````(`0"````````````````````!``D$``"@`0``
M```````````````````!``$````X`0"````````````````````!``D$``"P
M`0``T"$!`&@%`````````````#@G`0"H"`````````````#@+P$`:`4`````
M````````2#4!`*@(`````````````/`]`0`^```````````````P/@$`6`(`
M````````````B$`!`-H%``````````````8`5P!)`$X`10!8`$4````H````
M$````"`````!``@````````!```````````````!`````0``!@8&``0$!``"
M`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S<P"?GY\`6%A8`+*R
ML@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.`$9&1@#+R\L`"PL+
M`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`[N[N`/3T]`#CX^,`
MY>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")B8D`BHJ*`#0T-`#H
MZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/CP"^OKX`;6UM``D)
M"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0`#P\/`!D9&0`PL+"
M`#(R,@`("`@`#0T-``\/#P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````%!04%!04%!04%!04%!00$!`35`
M(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!%0$!)A4!`0\!`0$!
M`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!`0$!$`H@(A45+B$N
M`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!`0%!$!`0$!`0$!`0
M$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0!P$!`0$@$!`0`0\0
M$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!!0\!`0$!`0$!`0$!
M`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#``#@`P``P`$``(`!
M``"``0``@`,``(`#````!P````<``(X/``#_'P``*````"````!``````0`(
M````````!````````````````0````$````````!`0$``@("``,#`P`$!`0`
M!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-#0T`#@X.``\/#P`0
M$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8&``9&1D`&AH:`!L;
M&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C`"0D)``E)24`)B8F
M`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`+R\O`#`P,``Q,3$`
M,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z.CH`.SL[`#P\/``]
M/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%10!&1D8`1T='`$A(
M2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0`%%140!24E(`4U-3
M`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`7%Q<`%U=70!>7EX`
M7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G9V<`:&AH`&EI:0!J
M:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R<@!S<W,`='1T`'5U
M=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]`'Y^?@!_?W\`@("`
M`(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`B8F)`(J*B@"+BXL`
MC(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4E)0`E965`):6E@"7
MEY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?GP"@H*``H:&A`**B
MH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ`*NKJP"LK*P`K:VM
M`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`MK:V`+>WMP"XN+@`
MN;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!P<$`PL+"`,/#PP#$
MQ,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,S`#-S<T`SL[.`,_/
MSP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7`-C8V`#9V=D`VMK:
M`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`X^/C`.3DY`#EY>4`
MYN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N[NX`[^_O`/#P\`#Q
M\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY^0#Z^OH`^_O[`/S\
M_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\/#P\/#PL#`0$!`0$
M!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$9,SL]/3T_/3\]/3T
M_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T]/3T]/3T]/3T[.R\
M-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT45%1400$9+SD[.RL
MC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$492L9.2D3#0L7,3,
M4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QTK-3DW+PT!`0$!`0$
M9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1DO-SDY.3DY-R,?)RT
M?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,S#R49,R\C%'<U+0T
M!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$!`0$!`1<M-3<W.3<
MW'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<=%'T=(3LS&3D?&RD
MM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RLA%&\K#0$!`0$!`1<
MM,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RLS,QLG/3\_/S\_/S\
M_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\_/S\_/SLS*Q1K#0$
M!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$!`0$!%RLM$S4_/ST
MM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RTG/3\_/S\_/S\_/24
M4<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RLS+RD-`0$!`0$!%RD
M9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1LC.3T_/R<4<Q1E.S\
M_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T_.11O,S,Q+R<-`0$
M!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$!`0$7)R\Q,3$Q,3$
MQ,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$Q,3$Q,3$Q,3$Q,3$
MO+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\M)PL!`0$!`0$7)RT
MQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<G)24E)24E)24E)24
ME)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%0T!`0$
M!`0$````/P```!\````/````!P````,````!`````````#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\H````
M$````"`````!``@````````!```````````````!`````0``!@8&``0$!``"
M`@(``````$)"0@"=G9T`;FYN`(&!@0"QL;$`____`'-S<P"?GY\`6%A8`+*R
ML@#]_?T`:VMK`/S\_`#7U]<`%A86`-K:V@"6EI8`CHZ.`$9&1@#+R\L`"PL+
M`*"@H`#^_OX`R<G)`,'!P0`#`P,`$1$1`).3DP"SL[,`[N[N`/3T]`#CX^,`
MY>7E`**BH@"ZNKH`KZ^O``<'!P!Q<7$`M;6U`/7U]0")B8D`BHJ*`#0T-`#H
MZ.@`-34U`%I:6@"WM[<`V-C8`-#0T``='1T`#`P,`(^/CP"^OKX`;6UM``D)
M"0";FYL`+R\O`,W-S0#9V=D`&1D9`$U-30!$1$0`4%!0`#P\/`!D9&0`PL+"
M`#(R,@`("`@`#0T-``\/#P``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````%!04%!04%!04%!04%!00$!`35`
M(P5!,4-!041%+@$!`0$!`0$%`3P7%P$!$P$!`0$!`0$!%0$!)A4!`0\!`0$!
M`0$!`1,!$0$%-0$F`0$!`0$!`0$B$A`Q#P4!(`$!`0$!`0$!$`H@(A45+B$N
M`0$!`0$?("$0(A`C(R4F)P$!`0$E$!`0$!`0$!`0%T4!`0%!$!`0$!`0$!`0
M$!`7`0$!#Q`0%1`0$!`0$!`0`0$!`1(0$!,?$!`0$!`0!P$!`0$@$!`0`0\0
M$!`0$0$!`0$!!R`0"@$!!1`0$#$!`0$!`0$!`0$!`0%!!0\!`0$!`0$!`0$!
M`0$!`0$!`0$!`0&``0``P`$``.`#``#P`P``\`,``/`#``#@`P``P`$``(`!
M``"``0``@`,``(`#````!P````<``(X/``#_'P``*````"````!``````0`(
M````````!````````````````0````$````````!`0$``@("``,#`P`$!`0`
M!04%``8&!@`'!P<`"`@(``D)"0`*"@H`"PL+``P,#``-#0T`#@X.``\/#P`0
M$!``$1$1`!(2$@`3$Q,`%!04`!45%0`6%A8`%Q<7`!@8&``9&1D`&AH:`!L;
M&P`<'!P`'1T=`!X>'@`?'Q\`("`@`"$A(0`B(B(`(R,C`"0D)``E)24`)B8F
M`"<G)P`H*"@`*2DI`"HJ*@`K*RL`+"PL`"TM+0`N+BX`+R\O`#`P,``Q,3$`
M,C(R`#,S,P`T-#0`-34U`#8V-@`W-S<`.#@X`#DY.0`Z.CH`.SL[`#P\/``]
M/3T`/CX^`#\_/P!`0$``04%!`$)"0@!#0T,`1$1$`$5%10!&1D8`1T='`$A(
M2`!)24D`2DI*`$M+2P!,3$P`34U-`$Y.3@!/3T\`4%!0`%%140!24E(`4U-3
M`%145`!5554`5E96`%=75P!86%@`65E9`%I:6@!;6UL`7%Q<`%U=70!>7EX`
M7U]?`&!@8`!A86$`8F)B`&-C8P!D9&0`965E`&9F9@!G9V<`:&AH`&EI:0!J
M:FH`:VMK`&QL;`!M;6T`;FYN`&]O;P!P<'``<7%Q`')R<@!S<W,`='1T`'5U
M=0!V=G8`=W=W`'AX>`!Y>7D`>GIZ`'M[>P!\?'P`?7U]`'Y^?@!_?W\`@("`
M`(&!@0""@H(`@X.#`(2$A`"%A84`AH:&`(>'AP"(B(@`B8F)`(J*B@"+BXL`
MC(R,`(V-C0".CHX`CX^/`)"0D`"1D9$`DI*2`).3DP"4E)0`E965`):6E@"7
MEY<`F)B8`)F9F0":FIH`FYN;`)R<G`"=G9T`GIZ>`)^?GP"@H*``H:&A`**B
MH@"CHZ,`I*2D`*6EI0"FIJ8`IZ>G`*BHJ`"IJ:D`JJJJ`*NKJP"LK*P`K:VM
M`*ZNK@"OKZ\`L+"P`+&QL0"RLK(`L[.S`+2TM`"UM;4`MK:V`+>WMP"XN+@`
MN;FY`+JZN@"[N[L`O+R\`+V]O0"^OKX`O[^_`,#`P`#!P<$`PL+"`,/#PP#$
MQ,0`Q<7%`,;&Q@#'Q\<`R,C(`,G)R0#*RLH`R\O+`,S,S`#-S<T`SL[.`,_/
MSP#0T-``T='1`-+2T@#3T],`U-34`-75U0#6UM8`U]?7`-C8V`#9V=D`VMK:
M`-O;VP#<W-P`W=W=`-[>W@#?W]\`X.#@`.'AX0#BXN(`X^/C`.3DY`#EY>4`
MYN;F`.?GYP#HZ.@`Z>GI`.KJZ@#KZ^L`[.SL`.WM[0#N[NX`[^_O`/#P\`#Q
M\?$`\O+R`//S\P#T]/0`]?7U`/;V]@#W]_<`^/CX`/GY^0#Z^OH`^_O[`/S\
M_`#]_?T`_O[^`/___P`T-#0\/#P\/#P\/#P\/#P\/#P\/#P\/#PL#`0$!`0$
M!%RTU-34U-34U-34U-S4U-34U-34U-34U+0T400$!`0$9,SL]/3T_/3\]/3T
M_/3\]/ST_/S\]/3LQ#11400$!`1DQ.3L]/3T]/3T]/3T]/3T]/3T]/3T[.R\
M-%%1400$!&3$Y.SL[.3L[.SDY.3D[.SD[.3L[.SLY+PT45%1400$9+SD[.RL
MC*S4Q)24G)2,C)24E)R<G.S<O#1145%1401DO-SD[.3$492L9.2D3#0L7,3,
M4;34Y-R\-%%145%1462\W.3D[.3D481TY(QLI)QLC+QTK-3DW+PT!`0$!`0$
M9+S<Y.3LY.2\48S4U(RLS'3,W*11Q.3<O#0$!`0$!`1DO-SDY.3DY-R,?)RT
M?(2<;,SDO'3$Y-RT-`0$!`0$!%R\W-SDY.3DU)RD7)2,S#R49,R\C%'<U+0T
M!`0$!`0$9+34W-S<Y-R,C*PLO+3,6L1DG&R4I-S4M#0$!`0$!`1<M-3<W.3<
MW'24U$RD](P\U'24=)Q,W,RT-`0$!`0$!&2TU-S<W.3<=%'T=(3LS&3D?&RD
MM&RDU+0T!`0$!`0$9+3,W-S,?(R$M/S,Q/STM/R\9,RLA%&\K#0$!`0$!`1<
MM,S4S(RDW-SL_/S\_/ST_.RT[*Q\E%&L-`0$!`0$!%RLS,QLG/3\_/S\_/S\
M_/S\_/S\Q)2,4:PT!`0$!`0$7*S$O'SD_/S\_/S\_/S\_/S\_/SLS*Q1K#0$
M!`0$!`1<K+1,Q/S\_.ST_/S\_/S\_/S\_/2\4<2L-`0$!`0$!%RLM$S4_/ST
MM-S\_/S\_/S\_/S\S%',Q*0T!`0$!`0$7*S$I%'\_/RTG/3\_/S\_/S\_/24
M4<S$I"P$!`0$!`1<I*11A.S\_,QDM.S\_/S\_/S\U&RLS+RD-`0$!`0$!%RD
M9'2\]/S\Q%&$G.3\_/S\_.2,4<S,Q*0T!`0$!`0$7*1LC.3T_/R<4<Q1E.S\
M_/S\O'3,S,2\G"P$!`0$!`1<I*QD7&2DI%&LS,1TE.3T_.11O,S,Q+R<-`0$
M!`0$!%RDM+R\C%%1Q,3,S*Q1?'R4;'R\S,3$O)PL!`0$!`0$7)R\Q,3$Q,3$
MQ,3$Q,1145%1O,3$Q,2TG#0$!`0$!`14G+3$Q,3$Q,3$Q,3$Q,3$Q,3$Q,3$
MO+2<+`0$!`0$!%2<O,3$O+R\O+R\O+R\O+R\O+R\O+R\M)PL!`0$!`0$7)RT
MQ+RTM+2TM+2TM+2TM+2TM+2TM+2LG"P$!`0$!`14A)2<G)24E)24E)24E)24
ME)24E)24E)2$)`0$!`0$!$147%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%Q<7%0T!`0$
M!`0$````/P```!\````/````!P````,````!`````````#\````_````/P``
M`#\````_````/P```#\````_````/P```#\````_````/P```#\````_````
M/P```#\````_````/P```#\````_````/P```#\````_````/P```#\```$`
M!``0$````0`(`&@%```!`"`@```!``@`J`@```(`$!````$`"`!H!0```P`@
M(````0`(`*@(```$````6`(T````5@!3`%\`5@!%`%(`4P!)`$\`3@!?`$D`
M3@!&`$\``````+T$[_X```$``````````````````````#\`````````!``$
M``$```````````````````"X`0```0!3`'0`<@!I`&X`9P!&`&D`;`!E`$D`
M;@!F`&\```"4`0```0`P`#``,``P`#``-`!"`#`````D``(``0!#`&\`;0!P
M`&$`;@!Y`$X`80!M`&4``````"`````L``(``0!&`&D`;`!E`$0`90!S`&,`
M<@!I`'``=`!I`&\`;@``````(````#``"``!`$8`:0!L`&4`5@!E`'(`<P!I
M`&\`;@``````,``N`#``+@`P`"X`,````"0``@`!`$D`;@!T`&4`<@!N`&$`
M;`!.`&$`;0!E````(````"@``@`!`$P`90!G`&$`;`!#`&\`<`!Y`'(`:0!G
M`&@`=````"`````L``(``0!,`&4`9P!A`&P`5`!R`&$`9`!E`&T`80!R`&L`
M<P``````(````"P``@`!`$\`<@!I`&<`:0!N`&$`;`!&`&D`;`!E`&X`80!M
M`&4````@````)``"``$`4`!R`&\`9`!U`&,`=`!.`&$`;0!E```````@````
M-``(``$`4`!R`&\`9`!U`&,`=`!6`&4`<@!S`&D`;P!N````,``N`#``+@`P
M`"X`,````$0````!`%8`80!R`$8`:0!L`&4`20!N`&8`;P``````)``$````
M5`!R`&$`;@!S`&P`80!T`&D`;P!N`````````+`$/#]X;6P@=F5R<VEO;CTB
M,2XP(B!E;F-O9&EN9STB551&+3@B('-T86YD86QO;F4](GEE<R(_/@T*/&%S
M<V5M8FQY('AM;&YS/2)U<FXZ<V-H96UA<RUM:6-R;W-O9G0M8V]M.F%S;2YV
M,2(@;6%N:69E<W1697)S:6]N/2(Q+C`B/@T*("`@(#QA<W-E;6)L>4ED96YT
M:71Y('9E<G-I;VX](C`N,"XP+C`B('1Y<&4](G=I;C,R(B!N86UE/2)005(M
M4&%C:V5R+D%P<&QI8V%T:6]N(B`O/@T*("`@(#QD97-C<FEP=&EO;CY005(M
M4&%C:V5R($%P<&QI8V%T:6]N/"]D97-C<FEP=&EO;CX-"B`@("`\9&5P96YD
M96YC>3X-"B`@("`@("`@/&1E<&5N9&5N=$%S<V5M8FQY/@T*("`@("`@("`@
M("`@/&%S<V5M8FQY261E;G1I='D@='EP93TB=VEN,S(B(&YA;64](DUI8W)O
M<V]F="Y7:6YD;W=S+D-O;6UO;BU#;VYT<F]L<R(@=F5R<VEO;CTB-BXP+C`N
M,"(@<'5B;&EC2V5Y5&]K96X](C8U.35B-C0Q-#1C8V8Q9&8B(&QA;F=U86=E
M/2(J(B!P<F]C97-S;W)!<F-H:71E8W1U<F4](BHB("\^#0H@("`@("`@(#PO
M9&5P96YD96YT07-S96UB;'D^#0H@("`@/"]D97!E;F1E;F-Y/@T*("`@(#QT
M<G5S=$EN9F\@>&UL;G,](G5R;CIS8VAE;6%S+6UI8W)O<V]F="UC;VTZ87-M
M+G8S(CX-"B`@("`@("`@/'-E8W5R:71Y/@T*("`@("`@("`@("`@/')E<75E
M<W1E9%!R:79I;&5G97,^#0H@("`@("`@("`@("`@("`@/')E<75E<W1E9$5X
M96-U=&EO;DQE=F5L(&QE=F5L/2)A<TEN=F]K97(B('5I06-C97-S/2)F86QS
M92(@+SX-"B`@("`@("`@("`@(#PO<F5Q=65S=&5D4')I=FEL96=E<SX-"B`@
M("`@("`@/"]S96-U<FET>3X-"B`@("`\+W1R=7-T26YF;SX-"B`@("`\8V]M
M<&%T:6)I;&ET>2!X;6QN<STB=7)N.G-C:&5M87,M;6EC<F]S;V9T+6-O;3IC
M;VUP871I8FEL:71Y+G8Q(CX-"B`@("`@("`@/&%P<&QI8V%T:6]N/@T*("`@
M("`@("`@("`@/"$M+2!7:6YD;W=S(#@N,2`M+3X-"B`@("`@("`@("`@(#QS
M=7!P;W)T961/4R!)9#TB>S%F-C<V8S<V+3@P93$M-#(S.2TY-6)B+3@S9#!F
M-F0P9&$W.'TB+SX-"B`@("`@("`@("`@(#PA+2T@5VEN9&]W<R!6:7-T82`M
M+3X-"B`@("`@("`@("`@(#QS=7!P;W)T961/4R!)9#TB>V4R,#$Q-#4W+3$U
M-#8M-#-C-2UA-69E+3`P.&1E964S9#-F,'TB+SX-"B`@("`@("`@("`@(#PA
M+2T@5VEN9&]W<R`W("TM/@T*("`@("`@("`@("`@/'-U<'!O<G1E9$]3($ED
M/2)[,S4Q,SAB.6$M-60Y-BTT9F)D+3AE,F0M83(T-#`R,C5F.3-A?2(O/@T*
M("`@("`@("`@("`@/"$M+2!7:6YD;W=S(#@@+2T^#0H@("`@("`@("`@("`\
M<W5P<&]R=&5D3U,@260](GLT83)F,CAE,RTU,V(Y+30T-#$M8F$Y8RUD-CED
M-&$T839E,SA](B\^#0H@("`@("`@("`@("`\(2TM(%=I;F1O=W,@,3`@+2T^
M#0H@("`@("`@("`@("`\<W5P<&]R=&5D3U,@260](GLX93!F-V$Q,BUB9F(S
M+31F93@M8CEA-2TT.&9D-3!A,35A.6%](B\^#0H@("`@("`@(#PO87!P;&EC
M871I;VX^#0H@("`@/"]C;VUP871I8FEL:71Y/@T*/"]A<W-E;6)L>3X-"@``
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````4```
M#````/BB````8```#````""@````L```&````-"@X*#PH`"A$*$@H3"A````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
-````````````````````
