/* compress.c -- LZO packer

   This file is part of the LZO real-time data compression package.

   Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
   Copyright (C) 1997 Markus Franz Xaver Johannes Oberhumer
   Copyright (C) 1996 Markus Franz Xaver Johannes Oberhumer

   The LZO library and packer is free software; you can redistribute it
   and/or modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; see the file COPYING.
   If not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   Markus F.X.J. Oberhumer
   markus.oberhumer@jk.uni-linz.ac.at
 */


#include "lzopack.h"


/*************************************************************************
//
**************************************************************************/

int x_set_method(int m, int l)
{
	int r;

	r = lzop_set_method(m,l);
	if (r >= 0)
		return r;
#if defined(USE_ZLIB)
	r = zlib_set_method(m,l);
	if (r >= 0)
		return r;
#endif

	return -1;
}


int x_get_method(header_t *h)
{
	int r;

	r = lzop_get_method(h);
	if (r >= 0)
		return r;
#if defined(USE_ZLIB)
	r = zlib_get_method(h);
	if (r >= 0)
		return 0;
#endif

	return 14;
}


/*************************************************************************
// memory setup
**************************************************************************/

lzo_bool x_enter(const header_t *h)
{
	if (h == NULL)
	{
		lzo_bool ok = 1;
		ok &= lzop_enter(h);
#if defined(USE_ZLIB)
		ok &= zlib_enter(h);
#endif
		return ok;
	}

	switch (h->method)
	{
	case M_LZO1X_1:
	case M_LZO1X_1_15:
	case M_LZO1X_999:
		return lzop_enter(h);
#if defined(USE_ZLIB)
	case M_ZLIB:
		return zlib_enter(h);
#endif
	}
	return 0;
}


void x_leave(const header_t *h)
{
	if (h == NULL)
	{
#if defined(USE_ZLIB)
		zlib_leave(h);
#endif
		lzop_leave(h);
		return;
	}

	switch (h->method)
	{
	case M_LZO1X_1:
	case M_LZO1X_1_15:
	case M_LZO1X_999:
		lzop_leave(h);
		break;
#if defined(USE_ZLIB)
	case M_ZLIB:
		zlib_leave(h);
		break;
#endif
	}
}



/*************************************************************************
// compress a file
**************************************************************************/

lzo_bool x_compress(file_t *fip, file_t *fop, header_t *h)
{
	lzo_bool ok;

	init_compress_header(h,fip,fop);
	switch (h->method)
	{
	case M_LZO1X_1:
	case M_LZO1X_1_15:
	case M_LZO1X_999:
		lzop_init_compress_header(h);
		break;
#if defined(USE_ZLIB)
	case M_ZLIB:
		zlib_init_compress_header(h);
		break;
#endif
	}

	if (!x_enter(h))
		e_memory();

	if (opt_verbose > 1)
	{
		if (opt_unlink)
			fprintf(con_term,"replacing %s with %s", fip->name, fop->name);
		else
			fprintf(con_term,"compressing %s into %s", fip->name, fop->name);
		fflush(con_term);
		set_err_nl(1);
	}

	write_header(fop,h);

	fip->bytes_processed = fop->bytes_processed = 0;

	switch (h->method)
	{
	case M_LZO1X_1:
	case M_LZO1X_1_15:
	case M_LZO1X_999:
		ok = lzop_compress(fip,fop,h);
		break;
#if defined(USE_ZLIB)
	case M_ZLIB:
		ok = zlib_compress(fip,fop,h);
		break;
#endif
	default:
		fatal(fip,"Internal error");
		ok = 0;
		break;
	}

	if (opt_cmd == CMD_COMPRESS && opt_verbose > 1)
	{
		fprintf(con_term, ok ? "\n" : " FAILED\n");
		fflush(con_term);
	}
	set_err_nl(0);

	x_leave(h);
	return ok;
}


/*************************************************************************
// decompress a file
**************************************************************************/

lzo_bool x_decompress(file_t *fip, file_t *fop,
                      const header_t *h, lzo_bool skip)
{
	lzo_bool ok;

	if (!x_enter(h))
		e_memory();

	if (skip)
	{
		assert(opt_cmd != CMD_TEST);
		assert(fop->fd < 0);
		if (opt_cmd == CMD_DECOMPRESS && opt_verbose > 0)
			fprintf(con_term,"skipping %s [%s]", fip->name, fop->name);
		fflush(con_term);
		set_err_nl(1);
	}
	else if (opt_cmd == CMD_DECOMPRESS && opt_verbose > 1)
	{
		if (opt_unlink)
			fprintf(con_term,"restoring %s into %s", fip->name, fop->name);
		else
			fprintf(con_term,"decompressing %s into %s", fip->name, fop->name);
		fflush(con_term);
		set_err_nl(1);
	}
	else if (opt_cmd == CMD_TEST && opt_verbose > 0)
	{
		/* note: gzip is quiet by default when testing, lzop is not */
		if (opt_verbose > 2)
			fprintf(con_term,"testing %s [%s]", fip->name, fop->name);
		else
			fprintf(con_term,"testing %s", fip->name);
		fflush(con_term);
		set_err_nl(1);
	}

	fip->bytes_processed = fop->bytes_processed = 0;

	switch (h->method)
	{
	case M_LZO1X_1:
	case M_LZO1X_1_15:
	case M_LZO1X_999:
		ok = lzop_decompress(fip,fop,h,skip);
		break;
#if defined(USE_ZLIB)
	case M_ZLIB:
		ok = zlib_decompress(fip,fop,h,skip);
		break;
#endif
	default:
		fatal(fip,"Internal error");
		ok = 0;
		break;
	}

	if (skip && opt_cmd == CMD_DECOMPRESS && opt_verbose > 0)
	{
		fprintf(con_term, ok ? "\n" : " FAILED\n");
		fflush(con_term);
	}
	else if (opt_cmd == CMD_DECOMPRESS && opt_verbose > 1)
	{
		fprintf(con_term, ok ? "\n" : " FAILED\n");
		fflush(con_term);
	}
	else if (opt_cmd == CMD_TEST && opt_verbose > 0)
	{
		fprintf(con_term, ok ? " OK\n" : " FAILED\n");
		fflush(con_term);
	}
	set_err_nl(0);

	if (ok && opt_cmd == CMD_LIST)
		do_list(h,fop->bytes_processed,fip->bytes_processed);
	if (ok && opt_cmd == CMD_INFO)
		do_info(h,fop->bytes_processed,fip->bytes_processed);

	x_leave(h);
	return ok;
}


/*************************************************************************
//
**************************************************************************/

void x_filter(lzo_byte *p, lzo_uint l, const header_t *h)
{
	unsigned f = (unsigned) h->filter;
	lzo_bool c = (opt_cmd == CMD_COMPRESS);

	if (f == 0 || l <= 0)
		return;
	if (f == 1)
	{
		if (c) t_sub1(p,l); else t_add1(p,l);
	}
	else if (f <= 16)
	{
		if (c) t_sub(p,l,f); else t_add(p,l,f);
	}
	else
	{
		fatal(NULL,"Invalid filter");
	}
}



/*
vi:ts=4
*/

