/*
** pnmtotiff.c - converts a portable anymap to a Tagged Image File
**
** Derived by Jef Poskanzer from ras2tif.c, which is:
**
** Copyright (c) 1990 by Sun Microsystems, Inc.
**
** Author: Patrick J. Naughton
** naughton@wind.sun.com
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation.
**
** This file is provided AS IS with no warranties of any kind.  The author
** shall have no liability with respect to the infringement of copyrights,
** trade secrets or any patents by this file or any part thereof.  In no
** event will the author be liable for any lost revenue or profits or
** other special, indirect and consequential damages.
*/

#include "pnm.h"
#include <fcntl.h>
#ifdef VMS
#ifdef SYSV
#undef SYSV
#endif
#include <tiffioP.h>
#endif
#include <tiffio.h>

#include "ppmcmap.h"
#define MAXCOLORS 256

struct cmdline_info {
    /* All the information the user supplied in the command line,
       in a form easy for the program to use.
    */
    char *input_filespec;  /* Filespecs of input files */
    int compression;   /* COMPRESSION Tiff tag value */
    int fillorder;     /* FILLORDER Tiff tag value */
    int g3options;     /* G3OPTIONS Tiff tag value or 0 for none */
    int predictor;     /* PREDICTOR Tiff tag value or -1 for none */
    int rowsperstrip;  /* -1 = unspecified */
    int minisblack;    /* logical: User wants MINISBLACK photometric */
    float xresolution; /* XRESOLUTION Tiff tag value or -1 for none */
    float yresolution; /* YRESOLUTION Tiff tag value or -1 for none */
};



static void
parse_command_line(int argc, char ** argv,
                   struct cmdline_info *cmdline_p) {
/*----------------------------------------------------------------------------
   Note that the file spec array we return is stored in the storage that
   was passed to us as the argv array.
-----------------------------------------------------------------------------*/
    optStruct *option_def = malloc(100*sizeof(optStruct));
        /* Instructions to OptParseOptions2 on how to parse our options.
         */
    optStruct2 opt;

    int none, packbits, lzw, g3, g4, msb2lsb, lsb2msb, opt_2d, fill;

    unsigned int option_def_index;

    option_def_index = 0;   /* incremented by OPTENTRY */
    OPTENTRY(0, "none",         OPT_FLAG,   &none,                    0);
    OPTENTRY(0, "packbits",     OPT_FLAG,   &packbits,                0);
    OPTENTRY(0, "lzw",          OPT_FLAG,   &lzw,                     0);
    OPTENTRY(0, "g3",           OPT_FLAG,   &g3,                      0);
    OPTENTRY(0, "g4",           OPT_FLAG,   &g4,                      0);
    OPTENTRY(0, "msb2lsb",      OPT_FLAG,   &msb2lsb,                 0);
    OPTENTRY(0, "lsb2msb",      OPT_FLAG,   &lsb2msb,                 0);
    OPTENTRY(0, "2d",           OPT_FLAG,   &opt_2d,                  0);
    OPTENTRY(0, "fill",         OPT_FLAG,   &fill,                    0);
    OPTENTRY(0, "minisblack",   OPT_FLAG,   &cmdline_p->minisblack,   0);
    OPTENTRY(0, "mb",           OPT_FLAG,   &cmdline_p->minisblack,   0);
    OPTENTRY(0, "predictor",    OPT_UINT,   &cmdline_p->predictor,    0);
    OPTENTRY(0, "rowsperstrip", OPT_UINT,   &cmdline_p->rowsperstrip, 0);
    OPTENTRY(0, "xresolution",  OPT_FLOAT,  &cmdline_p->xresolution,  0);
    OPTENTRY(0, "yresolution",  OPT_FLOAT,  &cmdline_p->yresolution,  0);

    /* Set the defaults */
    none = packbits = lzw = g3 = g4 = msb2lsb = lsb2msb = opt_2d = FALSE;
    fill = FALSE;
    cmdline_p->predictor = -1;
    cmdline_p->rowsperstrip = -1;
    cmdline_p->minisblack = FALSE;
    cmdline_p->xresolution = -1;
    cmdline_p->yresolution = -1;

    opt.opt_table = option_def;
    opt.short_allowed = FALSE;  /* We have no short (old-fashioned) options */
    opt.allowNegNum = FALSE;  /* We have no parms that are negative numbers */

    pm_optParseOptions2(&argc, argv, opt, 0);
        /* Uses and sets argc, argv, and some of *cmdline_p and others. */

    if (none + packbits + lzw + g3 + g4 > 1)
        pm_error("You specified more than one compression option.  "
                 "Only one of -none, -packbits, -lze, -g3, and -g4 "
                 "is allowed.");
    
    if (none)
        cmdline_p->compression = COMPRESSION_NONE;
    else if (packbits)
        cmdline_p->compression = COMPRESSION_PACKBITS;
    else if (lzw)
        cmdline_p->compression = COMPRESSION_LZW;
    else if (g3)
        cmdline_p->compression = COMPRESSION_CCITTFAX3;
    else if (g4)
        cmdline_p->compression = COMPRESSION_CCITTFAX4;
    else
        cmdline_p->compression = COMPRESSION_NONE;
    
    if (msb2lsb + lsb2msb > 1)
        pm_error("You specified both -msb2lsb and -lsb2msb.  "
                 "These are conflicting options.");

    if (msb2lsb)
        cmdline_p->fillorder = FILLORDER_MSB2LSB;
    else if (lsb2msb)
        cmdline_p->fillorder = FILLORDER_LSB2MSB;
    else 
        cmdline_p->fillorder = FILLORDER_MSB2LSB;
    
    cmdline_p->g3options = 0;  /* initial value */
    if (opt_2d)
        cmdline_p->g3options |= GROUP3OPT_2DENCODING;
    if (fill)
        cmdline_p->g3options |= GROUP3OPT_FILLBITS;

    if (cmdline_p->predictor != -1 && 
        cmdline_p->predictor != 1 && cmdline_p->predictor != 2)
        pm_error("-predictor may be only 1 or 2.  You specified %d.", 
                 cmdline_p->predictor);

    if (cmdline_p->rowsperstrip != -1 && cmdline_p->rowsperstrip < 1)
        pm_error("-rowsperstrip must be positive.  You specified %d.",
                 cmdline_p->rowsperstrip);

    if (cmdline_p->xresolution != -1 && cmdline_p->xresolution < 1)
        pm_error("-xresolution must be positive.  You specified %d.",
                 cmdline_p->xresolution);

    if (cmdline_p->yresolution != -1 && cmdline_p->yresolution < 1)
        pm_error("-yresolution must be positive.  You specified %d.",
                 cmdline_p->yresolution);

    if (argc-1 == 0) 
        cmdline_p->input_filespec = "-";
    else if (argc-1 != 1)
        pm_error("Program takes zero or one argument (filename).  You "
                 "specified %d", argc-1);
    else
        cmdline_p->input_filespec = argv[1];

}



/* Note: PUTSAMPLE doesn't work if bitspersample is 1-4. */
#define PUTSAMPLE { \
    if ( maxval != tiff_maxval ) \
        s = s * tiff_maxval / maxval; \
    if (bitspersample > 8) \
        *tP++ = s >> 8; \
    *tP++ = s & 0xff; \
    }

int
main(int argc, char *argv[]) {
    struct cmdline_info cmdline;
    char* inf;
    FILE* ifp;
    const xel** xels;
    colorhist_vector chv;
    colorhash_table cht;
    unsigned short red[MAXCOLORS], grn[MAXCOLORS], blu[MAXCOLORS];
    int cols, rows, format, row, colors, i;
    register int col;
    xelval maxval;  /* maxval of PNM input */
    int grayscale;
    TIFF* tif;
    unsigned short photometric;
    unsigned short rowsperstrip;
    unsigned short samplesperpixel;
    unsigned short bitspersample;
    unsigned short tiff_maxval;
       /* This is the maxval of the samples in the tiff file.  It is 
          determined solely by the bits per sample ('bitspersample').
          */
    int bytesperrow;
    unsigned char* buf;
    unsigned char* tP;

    pnm_init( &argc, argv );

    parse_command_line(argc, argv, &cmdline);
    
    ifp = pm_openr(cmdline.input_filespec);

    if (strcmp(cmdline.input_filespec, "-") == 0)
        inf = "Standard Input";
    else 
        inf = cmdline.input_filespec;

    xels = (const xel **) pnm_readpnm( ifp, &cols, &rows, &maxval, &format );
    pm_close( ifp );

    /* Check for grayscale. */
    switch ( PNM_FORMAT_TYPE(format) )
	{
	case PPM_TYPE:
	pm_message( "computing colormap..." );
	chv = ppm_computecolorhist((xel **) xels, cols, rows, MAXCOLORS, &colors );
	if ( chv == (colorhist_vector) 0 )
	    {
	    pm_message(
		"Too many colors - proceeding to write a 24-bit RGB file." );
	    pm_message(
		"If you want an 8-bit palette file, try doing a 'ppmquant %d'.",
		MAXCOLORS );
	    grayscale = 0;
	    }
	else
	    {
	    pm_message( "%d colors found", colors );
	    grayscale = 1;
	    for ( i = 0; i < colors; ++i )
		{
		register xelval r, g, b;

		r = PPM_GETR( chv[i].color );
		g = PPM_GETG( chv[i].color );
		b = PPM_GETB( chv[i].color );
		if ( r != g || g != b )
		    {
		    grayscale = 0;
		    break;
		    }
		}
	    }
	break;

	default:
	chv = (colorhist_vector) 0;
	grayscale = 1;
	break;
	}

    /* Open output file. */
    fcntl( 1, F_SETFL, O_NONBLOCK ) ; 
      /* acooke dec99 - otherwise blocks on read inside 
         next line (Linux i386) 
         */
    tif = TIFFFdOpen( 1, "Standard Output", "w" );
    if ( tif == NULL )
	pm_error( "error opening standard output as TIFF file" );

    /* Figure out TIFF parameters. */

    /* We determine the bits per sample by the maxval, except that if
       it would be more than 8, we just use 16.  I don't know if bits
       per sample between 8 and 16 are legal, but they aren't very
       nice in any case.  If users want them, we should provide an
       option.  It is not clear why we don't use bits per pixel < 8
       for RGB images.  Note that code to handle maxvals <= 255 was
       written long before maxval > 255 was possible and there are
       backward compatibility requirements.  
    */
    switch ( PNM_FORMAT_TYPE(format) )
	{
	case PPM_TYPE:
	if ( chv == (colorhist_vector) 0 )
	    {
	    samplesperpixel = 3;
	    bitspersample = maxval > 255 ? 16 : 8;
	    photometric = PHOTOMETRIC_RGB;
	    }
	else if ( grayscale )
	    {
	    samplesperpixel = 1;
	    bitspersample = maxval > 255 ? 16 : pm_maxvaltobits( maxval );
	    photometric = PHOTOMETRIC_MINISBLACK;
	    }
	else
	    {
	    samplesperpixel = 1;
	    bitspersample = 8;
	    photometric = PHOTOMETRIC_PALETTE;
	    }
	break;

	case PGM_TYPE:
	samplesperpixel = 1;
	bitspersample = maxval > 255 ? 16 : pm_maxvaltobits( maxval );
	photometric = PHOTOMETRIC_MINISBLACK;
	break;

	default:
	samplesperpixel = 1;
    bitspersample = 1;
    if (((cmdline.compression == COMPRESSION_CCITTFAX3) ||
         (cmdline.compression == COMPRESSION_CCITTFAX4)) &&
        ! cmdline.minisblack)
        photometric = PHOTOMETRIC_MINISWHITE;
    else
        photometric = PHOTOMETRIC_MINISBLACK;
    break;
	}
    if (bitspersample < 8) {
        int samplesperbyte;
        samplesperbyte = 8 / bitspersample;
        bytesperrow = 
            (cols * samplesperpixel + samplesperbyte-1) / samplesperbyte;
    } else 
        bytesperrow = (cols * samplesperpixel * bitspersample) / 8;
    tiff_maxval = pm_bitstomaxval( bitspersample );

    if ( cmdline.rowsperstrip == -1 )
        rowsperstrip = ( 8 * 1024 ) / bytesperrow;
    else 
        rowsperstrip = cmdline.rowsperstrip;
    buf = (unsigned char*) malloc( bytesperrow );
    if ( buf == (unsigned char*) 0 )
	pm_error( "can't allocate memory for row buffer" );

    /* Set TIFF parameters. */
    TIFFSetField( tif, TIFFTAG_IMAGEWIDTH, cols );
    TIFFSetField( tif, TIFFTAG_IMAGELENGTH, rows );
    TIFFSetField( tif, TIFFTAG_BITSPERSAMPLE, bitspersample );
    TIFFSetField( tif, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT );
    TIFFSetField( tif, TIFFTAG_COMPRESSION, cmdline.compression );
    if (cmdline.compression == COMPRESSION_CCITTFAX3 && cmdline.g3options != 0)
        TIFFSetField( tif, TIFFTAG_GROUP3OPTIONS, cmdline.g3options );
    if ( cmdline.compression == COMPRESSION_LZW && cmdline.predictor != -1 )
        TIFFSetField( tif, TIFFTAG_PREDICTOR, cmdline.predictor );
    TIFFSetField( tif, TIFFTAG_PHOTOMETRIC, photometric );
    TIFFSetField( tif, TIFFTAG_FILLORDER, cmdline.fillorder );
    TIFFSetField( tif, TIFFTAG_DOCUMENTNAME, inf );
    TIFFSetField( tif, TIFFTAG_IMAGEDESCRIPTION, "converted PNM file" );
    TIFFSetField( tif, TIFFTAG_SAMPLESPERPIXEL, samplesperpixel );
    TIFFSetField( tif, TIFFTAG_ROWSPERSTRIP, rowsperstrip );
    if ( cmdline.xresolution != -1 )
		TIFFSetField( tif, TIFFTAG_XRESOLUTION, cmdline.xresolution );
    if ( cmdline.yresolution != -1 )
		TIFFSetField( tif, TIFFTAG_YRESOLUTION, cmdline.yresolution );
    TIFFSetField( tif, TIFFTAG_RESOLUTIONUNIT, RESUNIT_INCH);
    /* TIFFSetField( tif, TIFFTAG_STRIPBYTECOUNTS, rows / rowsperstrip ); */
    TIFFSetField( tif, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG );

    if ( chv == (colorhist_vector) 0 )
	cht = (colorhash_table) 0;
    else
	{
	/* Make TIFF colormap. */
	for ( i = 0; i < colors; ++i )
	    {
	    red[i] = (long) PPM_GETR( chv[i].color ) * 65535L / maxval;
	    grn[i] = (long) PPM_GETG( chv[i].color ) * 65535L / maxval;
	    blu[i] = (long) PPM_GETB( chv[i].color ) * 65535L / maxval;
	    }
	TIFFSetField( tif, TIFFTAG_COLORMAP, red, grn, blu );

	/* Convert color vector to color hash table, for fast lookup. */
	cht = ppm_colorhisttocolorhash( chv, colors );
	ppm_freecolorhist( chv );
	}

    /* Now write the TIFF data. */
    for ( row = 0; row < rows; ++row )
	{
	if ( PNM_FORMAT_TYPE(format) == PPM_TYPE && ! grayscale )
	    {
	    if ( cht == (colorhash_table) 0 )
		{
            /* It's an RGB raster */
		for ( col = 0, tP = buf; col < cols; ++col )
		    {
		    register xelval s;

		    s = PPM_GETR( xels[row][col] );
            PUTSAMPLE;
		    s = PPM_GETG( xels[row][col] );
            PUTSAMPLE;
		    s = PPM_GETB( xels[row][col] );
            PUTSAMPLE;
		    }
		}
	    else
		{
            /* It's a colormapped raster */
		for ( col = 0, tP = buf; col < cols; ++col )
		    {
		    register int s;

		    s = ppm_lookupcolor( cht, &xels[row][col] );
		    if ( s == -1 )
			pm_error(
			    "color not found?!?  row=%d col=%d  r=%d g=%d b=%d",
			    row, col, PPM_GETR( xels[row][col] ), 
                PPM_GETG( xels[row][col] ),
			    PPM_GETB( xels[row][col] ) );
		    *tP++ = (unsigned char) s;
		    }
		}
	    }
	else
	    {
            /* It's grayscale */
            if (bitspersample == 8 || bitspersample == 16) {
                unsigned char* tP;
                tP = buf;
                for (col = 0; col < cols; ++col) {
                    xelval s = PNM_GET1(xels[row][col]);
                    PUTSAMPLE;
                }
            } else {
                int col;
                int bitshift;
                unsigned char byte;
                xelval s;

                bitshift = 8 - bitspersample;
                byte = 0;
                for ( col = 0, tP = buf; col < cols; ++col )
                {
                    s = PNM_GET1( xels[row][col] );
                    if ( maxval != tiff_maxval )
                        s = (long) s * tiff_maxval / maxval;
 
                    if (photometric == PHOTOMETRIC_MINISWHITE)
                        s = tiff_maxval - s;
                    byte |= s << bitshift;
                    bitshift -= bitspersample;
                    if ( bitshift < 0 )
                    {
                        *tP++ = byte;
                        bitshift = 8 - bitspersample;
                        byte = 0;
                    }
                }
                if ( bitshift != 8 - bitspersample )
                    *tP++ = byte;
            }
        }

	if ( TIFFWriteScanline( tif, buf, row, 0 ) < 0 )
	    pm_error( "failed a scanline write on row %d", row );
	}
    TIFFFlushData( tif );
    TIFFClose( tif );

    exit( 0 );
    }

