//----------------------------------------------------------------------------
//
// AATEXT 1.0 beta - Antialiased text fonts for Allegro
//
// Douglas Eleveld (D.J.Eleveld@anest.azg.nl)
//
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
// Basic text drawing functions
//----------------------------------------------------------------------------
#include <stdio.h>
#include <stdarg.h>
#include "aatext.h"

/* Used for fast changing of the blender for the truecolor aliasing */
extern int _blender_alpha;

/* Used for the background color */
extern int _textmode;

/* Used for scratchpad for blitting the characters */
BITMAP *_scratchpad = NULL;

/* Used for fast ailiasing for filled background modes */
int _aliasmap_foreground = -1;
int _aliasmap_background = -1;
int _aliasmap_depth = -1;
int _aliasmap[8];

/* Used for fast changing of the transparency map in 256 color aliasing */
static COLOR_MAP color_map46;
static COLOR_MAP color_map141;

/*----------------------------------------------------------------------------
   Setup the global aliasmap for fast aliasing of text with
   filled backgrounds.
*/
static void _setup_aliasmap (int colour, int depth)
   {
   int cr, cg, cb, tr, tg, tb;
   int(*colmaker)(int,int,int);

   int darkness, index;

   switch(depth)
      {
      case(8):
         cr = getr8(colour);
         cg = getg8(colour);
         cb = getb8(colour);

         tr = getr8(_textmode);
         tg = getg8(_textmode);
         tb = getb8(_textmode);

         colmaker = makecol8;
         break;
      case(15):
         cr = getr15(colour);
         cg = getg15(colour);
         cb = getb15(colour);

         tr = getr15(_textmode);
         tg = getg15(_textmode);
         tb = getb15(_textmode);

         colmaker = makecol15;
         break;
      case(16):
         cr = getr16(colour);
         cg = getg16(colour);
         cb = getb16(colour);

         tr = getr16(_textmode);
         tg = getg16(_textmode);
         tb = getb16(_textmode);

         colmaker = makecol16;
         break;
      case(24):
         cr = getr24(colour);
         cg = getg24(colour);
         cb = getb24(colour);

         tr = getr24(_textmode);
         tg = getg24(_textmode);
         tb = getb24(_textmode);

         colmaker = makecol24;
         break;
      case(32):
         cr = getr32(colour);
         cg = getg32(colour);
         cb = getb32(colour);

         tr = getr32(_textmode);
         tg = getg32(_textmode);
         tb = getb32(_textmode);

         colmaker = makecol32;
         break;
      default:
         cr = getr(colour);
         cg = getg(colour);
         cb = getb(colour);

         tr = getr(_textmode);
         tg = getg(_textmode);
         tb = getb(_textmode);

         colmaker = makecol;
         break;
      }

   /* Build the color map */
   darkness = 37;
   for(index=1;index<7;index++)
      {
      _aliasmap[index] = colmaker((cr*darkness+(255-darkness)*tr)>>8,
                                  (cg*darkness+(255-darkness)*tg)>>8,
                                  (cb*darkness+(255-darkness)*tb)>>8);
      darkness += 37;
      }
   _aliasmap[0] = _textmode;
   _aliasmap[7] = colour;

   /* Confirm that the aliasmap has been built */
   _aliasmap_foreground = colour;
   _aliasmap_background = _textmode;
   _aliasmap_depth = depth;
   }

/*----------------------------------------------------------------------------
   Very fast 6 level antialiased text blitting for 8 bit modes
   and filled backgrounds.  I don't think it's possible to get
   it much faster than this...
*/
static void _antialiased_8bit_6level_filled_textout (BITMAP *bmp, FONT *font, char *string, int x, int y, int colour)
   {
   int i;
   BITMAP *b;
   FONT_PROP *fp = font->dat.dat_prop;

   /* Variables for bitmap construction */
   unsigned char* input;
   unsigned char* output;
   unsigned char* inputend;

   /* Possible setup the quick color alias map */
   if((colour!=_aliasmap_foreground)||
      (_textmode!=_aliasmap_background)||
      (_aliasmap_depth!=8))
      {
      _setup_aliasmap(colour,8);
      }

   /* Possible setup for internal scratchpad bitmap */
   if((_scratchpad==NULL)||
      (bitmap_color_depth(_scratchpad)!=8))
      {
      if(_scratchpad!=NULL) destroy_bitmap(_scratchpad);
      _scratchpad = create_bitmap_ex(8,30,30);
      }

   /* Blit the whole string */
   while(*string)
      {
      /* Find the characters bitmap */
      i = (int)*string - ' ';
      if(i < 0 || i > FONT_SIZE) i = 0;
      b = fp->dat[i];

      /* Build each character */
      for(i = 0; i < b->h; i++)
         {
         input = b->line[i];
         inputend = &b->line[i][b->w];
         output = _scratchpad->line[i];

         /* Super fast blit of a row to temp bitmap */
         while(input!=inputend)
            *output++ = _aliasmap[*input++>>5];
         }
      /* Blit the character to the destination */
      blit(_scratchpad,bmp,0,0,x,y,b->w,b->h);

      /* Goto the next character */
      x += b->w;
      string++;
      }
   }

/*----------------------------------------------------------------------------
   Very fast 6 level antialiased text blitting for true color modes
   and filled backgrounds.  I don't think it's possible to get
   it much faster than this...
*/
static void _antialiased_16bit_6level_filled_textout (BITMAP *bmp, FONT *font, char *string, int x, int y, int colour)
   {
   int i;
   BITMAP *b;
   FONT_PROP *fp = font->dat.dat_prop;

   /* Variables for bitmap construction */
   unsigned char* input;
   unsigned short* output;
   unsigned char* inputend;

   /* Possible setup the quick color alias map */
   if((colour!=_aliasmap_foreground)||
      (_textmode!=_aliasmap_background)||
      (_aliasmap_depth!=16))
      {
      _setup_aliasmap(colour,16);
      }

   /* Possible setup for internal scratchpad bitmap */
   if((_scratchpad==NULL)||
      (bitmap_color_depth(_scratchpad)!=16))
      {
      if(_scratchpad!=NULL) destroy_bitmap(_scratchpad);
      _scratchpad = create_bitmap_ex(16,30,30);
      }

   /* Blit the whole string */
   while(*string)
      {
      /* Find the characters bitmap */
      i = (int)*string - ' ';
      if(i < 0 || i > FONT_SIZE) i = 0;
      b = fp->dat[i];

      /* Build each character */
      for(i = 0; i < b->h; i++)
         {
         input = b->line[i];
         inputend = &b->line[i][b->w];
         output = (short*)_scratchpad->line[i];

         /* Super fast blit of a row to temp bitmap */
         while(input!=inputend)
            *output++ = _aliasmap[*input++>>5];
         }
      /* Blit the character to the destination */
      blit(_scratchpad,bmp,0,0,x,y,b->w,b->h);

      /* Goto the next character */
      x += b->w;
      string++;
      }
   }
/*----------------------------------------------------------------------------
   Very fast 6 level antialiased text blitting for true color modes
   and filled backgrounds.  I don't think it's possible to get
   it much faster than this...
*/
static void _antialiased_24bit_6level_filled_textout (BITMAP *bmp, FONT *font, char *string, int x, int y, int colour)
   {
   int i, c;
   BITMAP *b;
   FONT_PROP *fp = font->dat.dat_prop;

   /* Variables for bitmap construction */
   unsigned char* input;
   unsigned char* output;
   unsigned char* inputend;

   /* Possible setup the quick color alias map */
   if((colour!=_aliasmap_foreground)||
      (_textmode!=_aliasmap_background)||
      (_aliasmap_depth!=24))
      {
      _setup_aliasmap(colour,24);
      }

   /* Possible setup for internal scratchpad bitmap */
   if((_scratchpad==NULL)||
      (bitmap_color_depth(_scratchpad)!=24))
      {
      if(_scratchpad!=NULL) destroy_bitmap(_scratchpad);
      _scratchpad = create_bitmap_ex(24,30,30);
      }

   /* Blit the whole string */
   while(*string)
      {
      /* Find the characters bitmap */
      i = (int)*string - ' ';
      if(i < 0 || i > FONT_SIZE) i = 0;
      b = fp->dat[i];

      /* Build each character */
      for(i = 0; i < b->h; i++)
         {
         input = b->line[i];
         inputend = &b->line[i][b->w];
         output = _scratchpad->line[i];

         /* Super fast blit of a row to temp bitmap */
         while(input!=inputend)
            {
            c = _aliasmap[*input++>>5];

            *output++ = getr24(c);
            *output++ = getg24(c);
            *output++ = getb24(c);
            }
         }
      /* Blit the character to the destination */
      blit(_scratchpad,bmp,0,0,x,y,b->w,b->h);

      /* Goto the next character */
      x += b->w;
      string++;
      }
   }
/*----------------------------------------------------------------------------
   Very fast 6 level antialiased text blitting for true color modes
   and filled backgrounds.  I don't think it's possible to get
   it much faster than this...
*/
static void _antialiased_32bit_6level_filled_textout (BITMAP *bmp, FONT *font, char *string, int x, int y, int colour)
   {
   int i;
   BITMAP *b;
   FONT_PROP *fp = font->dat.dat_prop;

   /* Variables for bitmap construction */
   unsigned char* input;
   unsigned int* output;
   unsigned char* inputend;

   /* Possible setup the quick color alias map */
   if((colour!=_aliasmap_foreground)||
      (_textmode!=_aliasmap_background)||
      (_aliasmap_depth!=32))
      {
      _setup_aliasmap(colour,32);
      }

   /* Possible setup for internal scratchpad bitmap */
   if((_scratchpad==NULL)||
      (bitmap_color_depth(_scratchpad)!=32))
      {
      if(_scratchpad!=NULL) destroy_bitmap(_scratchpad);
      _scratchpad = create_bitmap_ex(32,30,30);
      }

   /* Blit the whole string */
   while(*string)
      {
      /* Find the characters bitmap */
      i = (int)*string - ' ';
      if(i < 0 || i > FONT_SIZE) i = 0;
      b = fp->dat[i];

      /* Build each character */
      for(i = 0; i < b->h; i++)
         {
         input = b->line[i];
         inputend = &b->line[i][b->w];
         output = (int*)_scratchpad->line[i];

         /* Super fast blit of a row to temp bitmap */
         while(input!=inputend)
            *output++ = _aliasmap[*input++>>5];
         }
      /* Blit the character to the destination */
      blit(_scratchpad,bmp,0,0,x,y,b->w,b->h);

      /* Goto the next character */
      x += b->w;
      string++;
      }
   }
/*----------------------------------------------------------------------------
   Pretty fast 6 level antialiased text blitting for empty backgrounds.
   It works in all color depths, although it may not be the fastest at
   each depth.
*/
static void _antialiased_6level_textout (BITMAP *bmp, FONT *font, char *string, int x, int y, int colour)
   {
   int xx, i, yy, xxend;
   BITMAP *b;
   FONT_PROP *fp = font->dat.dat_prop;

   unsigned char *input;

   /* Using address of labels for jump table for 256 color mode
      for use when drawing a filled background */
   void* jumptable_256[] = {
      &&done,
      &&label_256_1,
      &&label_256_2,
      &&label_256_3,
      &&label_256_4,
      &&label_256_5,
      &&label_256_6,
      &&label_256_6
      };

   /* Blit each character */
   while(*string)
      {
      /* Find the characters bitmap */
      i = (int)*string - ' ';
      if(i < 0 || i > FONT_SIZE) i = 0;
      b = fp->dat[i];

      /* Plot the aliased pixels */
      for(i = 0; i < b->h; i++)
         {
         input = b->line[i];

         yy = y+i;
         xxend = x + b->w;

         for(xx = x; xx < xxend; xx++)
            {
            /* Use of the jump table */
            goto *jumptable_256[*input++>>5];

            /* Plot once with a light color */
            label_256_1:
               drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
               color_map = &color_map46;
               putpixel(bmp,xx ,yy, colour);
               drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);
               goto done;

            /* Plot twice with a light color */
            label_256_2:
               drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
               color_map = &color_map46;
               putpixel(bmp,xx ,yy, colour);
               putpixel(bmp,xx ,yy, colour);
               drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);
               goto done;

            /* Plot once with a dark color */
            label_256_3:
               drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
               color_map = &color_map141;
               putpixel(bmp,xx ,yy, colour);
               drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);
               goto done;

            /* Plot once with a dark color and then once with a light color */
            label_256_4:
               drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
               color_map = &color_map141;
               putpixel(bmp,xx ,yy, colour);
               color_map = &color_map46;
               putpixel(bmp,xx ,yy, colour);
               drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);
               goto done;

            /* Plot twice with a dark color */
            label_256_5:
               drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);
               color_map = &color_map141;
               putpixel(bmp,xx ,yy, colour);
               putpixel(bmp,xx ,yy, colour);
               drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);
               goto done;

            /* Plot the full foreground color */
            label_256_6:
               putpixel(bmp,xx ,yy, colour);

            /* label for bieng done */
            done:
            }
         }
      x += b->w;
      string++;
      }
   }
/*----------------------------------------------------------------------------
   Pretty fast antialiased text blitting for truecolor modes and
   empty backgrounds. It fully supports 256 aliasing levels
*/
static void _antialiased_truecolor_textout (BITMAP *bmp, FONT *font, char *string, int x, int y, int colour)
   {
   int i, j;
   BITMAP *b;
   FONT_PROP *fp = font->dat.dat_prop;

   unsigned char *input;
   unsigned char *inputend;

   set_trans_blender(0, 0, 0, 128);
   drawing_mode(DRAW_MODE_TRANS, NULL, 0, 0);

   while(*string)
      {
      i = (int)*string - ' ';
      if(i < 0 || i > FONT_SIZE) i = 0;
      b = fp->dat[i];

      for(i = y; i < y+b->h; i++)
         {
         input = b->line[i-y];
         inputend = &b->line[i-y][b->w];

         j = x;

         /* Super fast blit of a row to temp bitmap */
         while(input!=inputend)
            {
            /* Set the translucency from the source character bitmap */
            _blender_alpha = *input++;

            /* Plot the text pixel with the correct translucency */
            if(_blender_alpha>0)
               putpixel(bmp, j, i, colour);

            j++;
            }
         }
      x += b->w;
      string++;
      }
   drawing_mode(DRAW_MODE_SOLID, NULL, 0, 0);
   }
/*----------------------------------------------------------------------------*/
/*
   General antialiased textout routines

   Setup the color maps for 256 color mode
*/

/* Cleanup functions */
static void _antialias_cleanup (void)
   {
   if(_scratchpad!=NULL) destroy_bitmap(_scratchpad);
   _scratchpad = NULL;
   }

void antialias_exit (void)
   {
   _antialias_cleanup();
   }

void antialias_init (PALETTE pal)
   {
   static int _exit_registered = 0;

	create_trans_table(&color_map46,pal, 46, 46, 46, NULL);
	create_trans_table(&color_map141,pal, 141, 141, 141, NULL);
   color_map = &color_map141;

   // Register the cleanup function, but only once
   if(_exit_registered==0) atexit(_antialias_cleanup);
   _exit_registered = 1;
   }

/*----------------------------------------------------------------------------
   Actual textout routine
*/
void aatextout (BITMAP *bmp, FONT *font, char *string, int x, int y, int colour)
   {
   /* Make sure that we haven't been called with a fixed width font by accident */
   if(font->height>0)
      {
      textout(bmp,font,string,x,y,colour);
      return;
      }

   /* Do we do a 256 color textout? */
   switch(bitmap_color_depth(bmp))
      {
      case(8):
         /* Use fast textout if we are using a filled background */
         if(_textmode>=0)
            _antialiased_8bit_6level_filled_textout(bmp,font,string,x,y,colour);
         /* Use general antialiased textout */
         else
            _antialiased_6level_textout(bmp,font,string,x,y,colour);
         break;
      case(15):
      case(16):
         /* Use general textout if we are using a filled background */
         if(_textmode>=0)
            _antialiased_16bit_6level_filled_textout(bmp,font,string,x,y,colour);

         /* Use general antialiased textout with full 255 aliasing levels */
         else
            _antialiased_truecolor_textout(bmp,font,string,x,y,colour);
         break;
      case(24):
         /* Use general textout if we are using a filled background */
         if(_textmode>=0)
            _antialiased_24bit_6level_filled_textout(bmp,font,string,x,y,colour);

         /* Use general antialiased textout with full 255 aliasing levels */
         else
            _antialiased_truecolor_textout(bmp,font,string,x,y,colour);
         break;
      case(32):
         /* Use general textout if we are using a filled background */
         if(_textmode>=0)
            _antialiased_32bit_6level_filled_textout(bmp,font,string,x,y,colour);

         /* Use general antialiased textout with full 255 aliasing levels */
         else
            _antialiased_truecolor_textout(bmp,font,string,x,y,colour);
         break;
      default:
      }
   }

/*----------------------------------------------------------------------------
   Centered textout routine
*/
void aatextout_center(BITMAP *bmp, FONT *font, char *string, int x, int y, int colour)
   {
   aatextout(bmp, font, string, x - (text_length(font, string) / 2), y, colour);
   }

//----------------------------------------------------------------------------
// these printf-like routines come almost directly from Allegro
//----------------------------------------------------------------------------
/* aatextprintf:
 *  Formatted text output, using a printf() style format string.
 */
void aatextprintf(BITMAP *bmp, FONT *f, int x, int y, int color, char *format, ...)
{
   char buf[256];

   va_list ap;
   va_start(ap, format);
   vsprintf(buf, format, ap);
   va_end(ap);

   aatextout(bmp, f, buf, x, y, color);
}

/* aatextprintf_center:
 *  Like aatextprintf(), but uses the x coordinate as the centre rather than
 *  the left of the string.
 */
void aatextprintf_center(BITMAP *bmp, FONT *f, int x, int y, int color, char *format, ...)
{
   char buf[256];

   va_list ap;
   va_start(ap, format);
   vsprintf(buf, format, ap);
   va_end(ap);

   aatextout_center(bmp, f, buf, x, y, color);
}


