/*
	Copyright (C) 2001 Achilleas Margaritis

	MX-Windows 2.0 window system for DJGPP + Allegro

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Library General Public
	License as published by the Free Software Foundation; either
	version 2 of the License, or (at your option) any later version.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	Library General Public License for more details.

	You should have received a copy of the GNU Library General Public
	License along with this library; if not, write to the Free
	Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	Contact : axilmar@b-online.gr
*/


#include "mxwin.h"


/*************************************************************************
	INTERNALS
 *************************************************************************/


#include <string.h>
#include <limits.h>
#include "_osdep.h"


/* checks if a rectangle is valid (facing the screen plane) */
#define _VALID_RECT(R)\
	(((R).left <= (R).right ) &&\
     ((R).top  <= (R).bottom))


/* invalidates a rectangle */
#define _INVALIDATE_RECT(R)\
	SET_RECT(R, SHRT_MAX, SHRT_MAX, SHRT_MIN, SHRT_MIN)


/* traverses a region */
#define _TRAVERSE_REGION(RGN, RECTVAR, INDEXVAR, START, CODE) {\
	register RECT *RECTVAR = (RGN)->rects + START;\
    register unsigned INDEXVAR;\
    register unsigned max##INDEXVAR = (RGN)->count;\
    for(INDEXVAR = START; INDEXVAR < max##INDEXVAR; INDEXVAR++) {\
    	CODE;\
        RECTVAR++;\
    }\
}


/* type casting */
#define _MK_WND(W)           ((_WINDOW *)W)
#define _WND                 _MK_WND(wnd)
#define _PARENT              _MK_WND(parent)
#define _MK_CUR(C)           ((_CURSOR *)C)
#define _CUR                 _MK_CUR(cur)


/* traverses children windows from top to bottom child */
#define _TRAVERSE_WINDOWS(WND, CODE) {\
    register _WINDOW *child = _MK_WND(WND)->top;\
    register _WINDOW *next;\
    while (child) {\
    	next = child->lower;\
        CODE;\
        child = next;\
    }\
}


/* checks if a window is drawable within a rectangle */
#define _IS_DRAWABLE(WND, RCT)\
	(_MK_WND(WND)->drawable && RECTS_OVERLAP(_MK_WND(WND)->clip, RCT))


/* calls a window proc */
#define _DO_PROC(WND, MSG, DATA)\
	((WND) ?\
     (_MK_WND(WND)->proc((WINDOW)WND, _MK_WND(WND)->data, MSG, DATA)) :\
     (0))


#define _GUI_BEAT            100
#define _MAX_EVENTS          100
#define _MAX_TIMERS          10


/* timer traversal */
#define _TRAVERSE_TIMERS(CODE) {\
	register _TIMER *tm = _timers + 1;\
    register int i;\
    for(i = 1; i < _MAX_TIMERS; i++) {\
    	CODE;\
    	tm++;\
    }\
}


/* region */
typedef struct _REGION {
	RECT *rects;
    unsigned alloc;
    unsigned count;
} _REGION;


/* internals of window */
typedef struct _WINDOW {
	int unused;
	struct _WINDOW *parent;
    struct _WINDOW *lower;
    struct _WINDOW *higher;
    struct _WINDOW *bottom;
    struct _WINDOW *top;
    unsigned count;
    short x;
    short y;
    USHORT width;
    USHORT height;
    RECT frame;
    RECT pos;
    RECT clip;
    RECT view;
    RECT clipview;
    _REGION fore_region;
    _REGION back_region;
    _REGION invalid_region;
    CURSOR cursor;
    WINDOW_PROC *proc;
    void *data;
    unsigned flags:30;
    BOOL drawable:1;
    BOOL drawn:1;
} _WINDOW;


/* an event queue */
typedef struct _QUEUE {
	unsigned free;
    unsigned avail;
    unsigned count;
    EVENT events[_MAX_EVENTS];
} _QUEUE;


/* timer */
typedef struct _TIMER {
	WINDOW wnd;
    unsigned timer;
    unsigned timeout;
} _TIMER;


/* cursor */
typedef struct _CURSOR {
	int unused;
    int x;
    int y;
    BITMAP *bitmap;
    BITMAP *current;
} _CURSOR;


/* forward declarations */
static BOOL _paint_loop_init(register _WINDOW *wnd, register RECT *rct,
	register BOOL overdraw_children);


/* internal variables */
static BOOL _paint = FALSE;
static RECT *_paint_rect = NULL;
static RECT _paint_clip;
static int _paint_count = 0;
static BOOL (*_paint_loop_proc)(register _WINDOW *, register RECT *,
	register BOOL) = _paint_loop_init;
static int _saved_cl = 0;
static int _saved_ct = 0;
static int _saved_cr = 0;
static int _saved_cb = 0;
static WINDOW _root = NULL;
static WINDOW _mouse = NULL;
static WINDOW _focus = NULL;
static WINDOW _grab = NULL;
static int _clicks = 0;
static int _click_timer = 0;
static int _click_timeout = 1000;
static int _button_down = 0;
static int _buttons[] = {0, 1, 2, 3};
static int _key_down[128];
static _QUEUE _raw_queue;
static _QUEUE _ready_queue;
static void (*_prev_mouse_callback)(int flags) = NULL;
static int (*_prev_keyboard_callback)(int k) = NULL;
static void (*_prev_keyboard_lowlevel_callback)(int k) = NULL;
static _TIMER _timers[_MAX_TIMERS];
static CURSOR _curr_cursor = NULL;


/* allocates an event */
static EVENT *_alloc_event(register _QUEUE *q)
{
	register EVENT *e;

    if (q->count == _MAX_EVENTS) return NULL;
    e = &q->events[q->free];
    q->free++;
    if (q->free == _MAX_EVENTS) q->free = 0;
    q->count++;
    return e;
}
END_OF_STATIC_FUNCTION(_alloc_event);


/* frees an event */
static EVENT *_free_event(register _QUEUE *q)
{
	register EVENT *e;

    if (!q->count) return NULL;
    e = &q->events[q->avail];
    q->avail++;
    if (q->avail == _MAX_EVENTS) q->avail = 0;
    q->count--;
    return e;
}
END_OF_STATIC_FUNCTION(_free_event);


/* records a timer event */
static void _record_timer(register int slot)
{
	register EVENT *e;

    _lock_events();
    e = _alloc_event(&_raw_queue);
    if (e) {
    	e->type = WE_TIMER;
        e->timer.slot = slot;
    }
    _unlock_events();
}
END_OF_STATIC_FUNCTION(_record_timer);


/* timer proc */
static void _timer_proc(void)
{
	/* handle clicks */
    _click_timer += _GUI_BEAT;
    if (_click_timer >= _click_timeout) {
    	_click_timer = 0;
        _clicks = 0;
    }

    /* handle timers */
    _TRAVERSE_TIMERS(
    	if (tm->wnd) {
        	tm->timer += _GUI_BEAT;
            if (tm->timer >= tm->timeout) {
            	tm->timer = 0;
                _record_timer(i);
            }
        }
    );
}
END_OF_STATIC_FUNCTION(_timer_proc);


/* records a mouse event */
static void _record_mouse(register int type)
{
	register EVENT *e;

    _lock_events();
    e = _alloc_event(&_raw_queue);
    if (e) {
    	e->type = type;
        e->mouse.x = mouse_x;
        e->mouse.y = mouse_y;
        e->mouse.button = _button_down;
        e->mouse.clicks = _clicks;
        e->mouse.shifts = key_shifts;
    }
    _unlock_events();
}
END_OF_STATIC_FUNCTION(_record_mouse);


/* mouse proc */
static void _mouse_proc(int flags)
{
	/* catch button press */
    if (mouse_b && !_button_down) {
    	_button_down = _buttons[mouse_b % 4];
        _record_mouse(WE_LBUTTONDOWN + _button_down - 1);
    }
    /* else catch button release */
    else if (!mouse_b && _button_down) {
    	_click_timer = 0;
        _clicks++;
        _record_mouse(WE_LBUTTONUP + _button_down - 1);
    	_button_down = 0;
    }

    /* catch mouse motion */
    if (flags & MOUSE_FLAG_MOVE) _record_mouse(WE_MOUSEMOVE);

    /* call the previous handler, if it exists */
    if (_prev_mouse_callback) _prev_mouse_callback(flags);
}
END_OF_STATIC_FUNCTION(_mouse_proc);


/* records a key event */
static void _record_key(register int type, register int k)
{
	register EVENT *e;

    _lock_events();
    e = _alloc_event(&_raw_queue);
    if (e) {
    	e->type = type;
        e->key.key = k;
        e->key.ascii = k & 255;
        e->key.scanc = k >> 8;
        e->key.shifts = key_shifts;
    }
    _unlock_events();
}
END_OF_STATIC_FUNCTION(_record_key);


/* key proc */
static int _key_proc(int k)
{
	/* record key press */
    _key_down[k >> 8] = k;
    _record_key(WE_KEYDOWN, k);

    /* call the previous handler, if it exists */
    if (_prev_keyboard_callback) return _prev_keyboard_callback(k);
	return k;
}
END_OF_STATIC_FUNCTION(_key_proc);


/* key lowlevel proc */
static void _key_lowlevel_proc(int k)
{
	register int scanc = k & 127;
    register int released = k & 128;

    /* react according to scan code */
    switch (scanc) {    
		/* catch modifier keys */
    	case KEY_LCONTROL:
        case KEY_RCONTROL:
        case KEY_LSHIFT:
        case KEY_RSHIFT:
        case KEY_ALT:
        case KEY_ALTGR:
        	_record_key(released ? WE_KEYUP : WE_KEYDOWN, scanc << 8);
        	break;

        /* for the rest of the keys, catch button up */
        default:
        	if (released) _record_key(WE_KEYUP, _key_down[scanc]);
            break;
    }

    /* call the previous handler, if it exists */
    if (_prev_keyboard_lowlevel_callback)
    	_prev_keyboard_lowlevel_callback(k);
}
END_OF_STATIC_FUNCTION(_key_lowlevel_proc);


/* installs the library */
static void _install()
{
	static BOOL _installed = FALSE;

    if (_installed) return;
    _installed = TRUE;
    install_keyboard();
    install_timer();
    install_mouse();
    LOCK_VARIABLE(_clicks);
    LOCK_VARIABLE(_click_timer);
    LOCK_VARIABLE(_click_timeout);
    LOCK_VARIABLE(_button_down);
    LOCK_VARIABLE(_buttons);
    LOCK_VARIABLE(_key_down);
    LOCK_VARIABLE(_raw_queue);
    LOCK_VARIABLE(_ready_queue);
	LOCK_VARIABLE(_prev_mouse_callback);
	LOCK_VARIABLE(_prev_keyboard_callback);
	LOCK_VARIABLE(_prev_keyboard_lowlevel_callback);
    LOCK_VARIABLE(_timers);
    LOCK_FUNCTION(_alloc_event);
    LOCK_FUNCTION(_free_event);
    LOCK_FUNCTION(_record_timer);
    LOCK_FUNCTION(_timer_proc);
    LOCK_FUNCTION(_record_mouse);
    LOCK_FUNCTION(_mouse_proc);
    LOCK_FUNCTION(_record_key);
    LOCK_FUNCTION(_key_proc);
    LOCK_FUNCTION(_key_lowlevel_proc);
    memset(&_raw_queue, 0, sizeof(_QUEUE));
    memset(&_ready_queue, 0, sizeof(_QUEUE));
    memset(&_timers, 0, sizeof(_timers));
	_prev_mouse_callback = mouse_callback;
	_prev_keyboard_callback = keyboard_callback;
	_prev_keyboard_lowlevel_callback = keyboard_lowlevel_callback;
    install_int(_timer_proc, _GUI_BEAT);
    mouse_callback = _mouse_proc;
    keyboard_callback = _key_proc;    
    keyboard_lowlevel_callback = _key_lowlevel_proc;
}


/* expands a region by the given amount of rectangles */
static BOOL _expand_region(register _REGION *rgn, register unsigned num)
{
	/* check if the region has enough room to accomodate the requested
       amount of rectangles */
	if (rgn->count + num < rgn->alloc) return TRUE;

    /* double the region root to accomodate the requested amount */
    rgn->alloc += MAX(MAX(rgn->alloc, num), 16);
    rgn->rects = realloc(rgn->rects, rgn->alloc * sizeof(RECT));

    /* check if reallocation succeeded */
    if (rgn->rects) return TRUE;

    /* since allocation failed, the region should not be left
       inconsistent */
    rgn->count = 0;
    rgn->alloc = 0;
    return FALSE;
}


/* splits a rectangle to 0..4 rectangles according to the way
   it is overlapped by another rectangle; it returns the
   number of rectangles created */
static int _split_rect(register RECT *s, register RECT *r, register RECT *res)
{
	register int n = 0;
    register short sl = s->left;
    register short st = s->top;
    register short sr = s->right;
    register short sb = s->bottom;
    register short rl = r->left;
    register short rt = r->top;
    register short rr = r->right;
    register short rb = r->bottom;
    register short maxt = MAX(st, rt);
    register short minb = MIN(sb, rb);

    /* split top */
    if (st < rt) {
    	SET_RECT(*res, sl, st, sr, maxt - 1);
        res++;
        n++;
    }

    /* split left */
    if (sl < rl) {
    	SET_RECT(*res, sl, maxt, rl - 1, minb);
        res++;
        n++;
    }

    /* split right */
    if (rr < sr) {
    	SET_RECT(*res, rr + 1, maxt, sr, minb);
        res++;
        n++;
    }

    /* split bottom */
    if (rb < sb) {
    	SET_RECT(*res, sl, minb + 1, sr, sb);
        res++;
        n++;
    }

    return n;
}


/* adds a rectangle to a region */
static INLINE BOOL _add_rect(register _REGION *rgn, register RECT *r)
{
	if (!_expand_region(rgn, 1)) return FALSE;
    *(rgn->rects + rgn->count) = *r;
    rgn->count++;
    return TRUE;
}


/* subtracts a rectangle from a region */
static BOOL _sub_rect(register _REGION *rgn, register RECT *r)
{
	_TRAVERSE_REGION(rgn, s, i, 0,
    	if RECTS_OVERLAP(*s, *r) {
        	if (!_expand_region(rgn, 4)) return FALSE;
            s = rgn->rects + i;
            rgn->count += _split_rect(s, r, rgn->rects + rgn->count);
            _INVALIDATE_RECT(*s);
        }
    );
    return TRUE;
}


/* clears a region */
static INLINE void _clear_region(register _REGION *rgn)
{
	if (rgn->rects) free(rgn->rects);
    memset(rgn, 0, sizeof(_REGION));
}


/* copies a region to another region */
static INLINE void _copy_region(register _REGION *src, register _REGION *dst)
{
	_clear_region(dst);
    if (!src->count) return;
    dst->rects = malloc(src->alloc * sizeof(RECT));
    if (!dst->rects) return;
    memcpy(dst->rects, src->rects, src->count * sizeof(RECT));
    dst->alloc = src->alloc;
    dst->count = src->count;
}


/* calculates the intersection of a region and a rectangle */
static void _clip_region(register _REGION *src, register RECT *r,
	register _REGION *dst)
{
	RECT t;

    dst->count = 0;
	_TRAVERSE_REGION(src, s, i, 0,
    	if _VALID_RECT(*s) {
        	if RECTS_OVERLAP(*s, *r) {
            	INTERSECT_RECTS(*s, *r, t);
                _add_rect(dst, &t);
            }
        }
    );
}


/* calculates the intersection between two regions */
static void _intersect_regions(register _REGION *a, register _REGION *b,
	register _REGION *dst)
{
	RECT t;

    dst->count = 0;
	_TRAVERSE_REGION(a, s, i, 0,
    	if _VALID_RECT(*s) _TRAVERSE_REGION(b, f, j, 0,
        	if RECTS_OVERLAP(*s, *f) {
            	INTERSECT_RECTS(*s, *f, t);
                _add_rect(dst, &t);
            }
        );
    );
}


/* optimizes a region by joining together rectangles that are aligned
   either horizontally or vertically and by removing invalid ones */
static INLINE void _optimize_region(register _REGION *rgn)
{
	register int invalid;

	if (!rgn->count) return;

    /* combine aligned rectangles */
    _TRAVERSE_REGION(rgn, s, i, 0,
    	if _VALID_RECT(*s) _TRAVERSE_REGION(rgn, f, j, i + 1,
        	if ((s->left == f->left) && (s->right == f->right)) {
            	if (s->bottom == f->top - 1) {
                	s->bottom = f->bottom;
                    _INVALIDATE_RECT(*f);
                }
                else if (s->top == f->bottom + 1) {
                	s->top = f->top;
                    _INVALIDATE_RECT(*f);
                }
            }
            else if ((s->top == f->top) && (s->bottom == f->bottom)) {
            	if (s->right == f->left - 1) {
                	s->right = f->right;
                    _INVALIDATE_RECT(*f);
                }
                else if (s->left == f->right + 1) {
                	s->left = f->left;
                    _INVALIDATE_RECT(*f);
                }
            }
        );
    );

    /* remove invalid rectangles */
    invalid = 0;
    _TRAVERSE_REGION(rgn, s, i, 0,
    	if _VALID_RECT(*s) {
        	if (invalid) *(s - invalid) = *s;
        }
        else invalid++;
    );
    rgn->count -= invalid;
}


/* subtracts region B from region A and puts the result in region DST */
static INLINE void _subtract_regions(register _REGION *a,
	register _REGION *b, register _REGION *dst)
{
	_copy_region(a, dst);
    _TRAVERSE_REGION(b, s, i, 0,
    	if _VALID_RECT(*s) _sub_rect(dst, s);
    );
}


/* returns TRUE if point in given region */
static INLINE BOOL _point_in_region(register _REGION *rgn,
	register short x, register short y)
{
    _TRAVERSE_REGION(rgn, s, i, 0,
    	if POINT_IN_RECT(*s, x, y) return TRUE;
    );
    return FALSE;
}


/* calculates the screen position, viewport and clipping of a window */
static INLINE void _calc_geometry(register _WINDOW *wnd)
{
	register _WINDOW *par = wnd->parent;

    if (par) {
    	wnd->pos.left   = wnd->x + par->view.left;
    	wnd->pos.top    = wnd->y + par->view.top;
        wnd->pos.right  = wnd->pos.left + wnd->width  - 1;
        wnd->pos.bottom = wnd->pos.top  + wnd->height - 1;
        INTERSECT_RECTS(wnd->pos, par->clipview, wnd->clip);
        wnd->drawable = (wnd->flags & WF_VISIBLE) && par->drawable &&
        	_VALID_RECT(wnd->clip);
    }
    else {
    	wnd->pos.left   = wnd->x;
    	wnd->pos.top    = wnd->y;
        wnd->pos.right  = wnd->pos.left + wnd->width  - 1;
        wnd->pos.bottom = wnd->pos.top  + wnd->height - 1;
        wnd->clip = wnd->pos;
        wnd->drawable = TRUE;
        wnd->flags |= WF_VISIBLE;
    }
    wnd->view.left    = wnd->pos.left   + wnd->frame.left;
    wnd->view.top     = wnd->pos.top    + wnd->frame.top;
    wnd->view.right   = wnd->pos.right  - wnd->frame.right;
    wnd->view.bottom  = wnd->pos.bottom - wnd->frame.bottom;
    INTERSECT_RECTS(wnd->view, wnd->clip, wnd->clipview);
}


/* subtracts a list of windows from a region */
static void _clip_wnd(register _REGION *rgn, register RECT *rct,
	register _WINDOW *wnd)
{
	while (wnd) {
    	if ((wnd->flags & WF_CLIPPED) && (_IS_DRAWABLE(wnd, *rct))) {
        	_sub_rect(rgn, &wnd->clip);
            if (!rgn->count) return;
        }
    	wnd = wnd->higher;
    }
}


/* calculates the foreground region of a window */
static void _calc_fore_region(register _WINDOW *wnd)
{
	register _WINDOW *par;
    
	wnd->fore_region.count = 0;
	if (!wnd->drawable) return;
    par = wnd->parent;
    if (par) {
    	_clip_region(&par->fore_region, &wnd->clip, &wnd->fore_region);
        if ((wnd->flags & WF_CLIPSIBLINGS) && (wnd->higher))
        	_clip_wnd(&wnd->fore_region, &wnd->clip, wnd->higher);
        _optimize_region(&wnd->fore_region);
    	return;
    }
	_add_rect(&wnd->fore_region, &wnd->clip);
}


/* calculates the background region of a window */
static void _calc_back_region(register _WINDOW *wnd)
{
	wnd->back_region.count = 0;
	if (!wnd->drawable) return;
    _copy_region(&wnd->fore_region, &wnd->back_region);
    if ((wnd->flags & WF_CLIPCHILDREN) && (wnd->bottom))
    	_clip_wnd(&wnd->back_region, &wnd->clip, wnd->bottom);
    _optimize_region(&wnd->back_region);
}


/* renders a window tree */
static void _render_windows(register _WINDOW *wnd)
{
	_calc_geometry(wnd);
    _calc_fore_region(wnd);
    _TRAVERSE_WINDOWS(wnd, _render_windows(child));
    _calc_back_region(wnd);
    wnd->drawn = TRUE;
}


/* unrenders a window tree */
static void _unrender_windows(register _WINDOW *wnd)
{
	wnd->fore_region.count = 0;
    wnd->back_region.count = 0;
    wnd->invalid_region.count = 0;
    wnd->drawable = FALSE;
    wnd->drawn = FALSE;
    _TRAVERSE_WINDOWS(wnd, _unrender_windows(child));
}


/* recalculates regions of all windows that fall within given rectangle */
static void _recalc_regions(register _WINDOW *wnd, register RECT *r)
{
    _calc_fore_region(wnd);
    _TRAVERSE_WINDOWS(wnd,
    	if _IS_DRAWABLE(child, *r) _recalc_regions(child, r);
    );
    _calc_back_region(wnd);
}


/* invalidates an area of a single window */
static INLINE void _invalidate_window(register _WINDOW *wnd, register RECT *r)
{
	wnd->invalid_region.count = 0;
    _clip_region(&wnd->back_region, r, &wnd->invalid_region);
    _optimize_region(&wnd->invalid_region);
}


/* invalidates given windows */
static void _invalidate_windows(register _WINDOW *wnd, register RECT *r)
{
    _TRAVERSE_WINDOWS(wnd,
    	if _IS_DRAWABLE(child, *r) _invalidate_windows(child, r);
    );
    _invalidate_window(wnd, r);
}


/* recalculates regions of all windows that fall within given rectangle
   and at the same time it invalidates the window area
   which falls into the rectangle */
static void _update_windows(register _WINDOW *wnd, register RECT *r)
{
    _calc_fore_region(wnd);
    _TRAVERSE_WINDOWS(wnd,
    	if _IS_DRAWABLE(child, *r) _update_windows(child, r);
    );
    _calc_back_region(wnd);
    _invalidate_window(wnd, r);
}


/* calculates the regions of windows that make up the
   background of another window (namely, the parent windows, the
   lower sibling windows and all their descentant windows) */
static void _calc_bg(register _WINDOW *par, register _WINDOW *child,
	register _WINDOW *to, register RECT *r)
{
	if (to) to = to->lower;
    while (child != to) {
    	if _IS_DRAWABLE(child, *r) _recalc_regions(child, r);
    	child = child->lower;
    }
    if (par) _calc_back_region(par);
}


/* invalidates the background windows */
static void _invalidate_bg(register _WINDOW *par, register _WINDOW *child,
	register _WINDOW *to, register RECT *r)
{
	if (to) to = to->lower;
    while (child != to) {
    	if _IS_DRAWABLE(child, *r) _invalidate_windows(child, r);
    	child = child->lower;
    }
    if (par) _invalidate_window(par, r);
}


/* updates the background windows */
static void _update_bg(register _WINDOW *par, register _WINDOW *child,
	register _WINDOW *to, register RECT *r)
{
	if (to) to = to->lower;
    while (child != to) {
    	if _IS_DRAWABLE(child, *r) _update_windows(child, r);
    	child = child->lower;
    }
    if (par) {
    	_calc_back_region(par);
        _invalidate_window(par, r);
    }
}


/* invalidates a region of a single window */
static INLINE void _invalidate_window_ex(register _WINDOW *wnd,
	register _REGION *rgn)
{
	wnd->invalid_region.count = 0;
    _intersect_regions(&wnd->back_region, rgn, &wnd->invalid_region);
    _optimize_region(&wnd->invalid_region);
}


/* invalidates given windows, but it uses a region as the invalid area */
static INLINE void _invalidate_windows_ex(register _WINDOW *wnd,
	register RECT *r, register _REGION *rgn)
{
    _TRAVERSE_WINDOWS(wnd,
    	if _IS_DRAWABLE(child, *r) _invalidate_windows_ex(child, r, rgn);
    );
    _invalidate_window_ex(wnd, rgn);
}


/* like '_update_windows' but uses a region for the invalid area */
static void _update_windows_ex(register _WINDOW *wnd, register RECT *r,
	register _REGION *rgn)
{
    _calc_fore_region(wnd);
    _TRAVERSE_WINDOWS(wnd,
    	if _IS_DRAWABLE(child, *r) _update_windows_ex(child, r, rgn);
    );
    _calc_back_region(wnd);
    _invalidate_window_ex(wnd, rgn);
}


/* updates the background windows, using a region for the invalid area */
static void _update_bg_ex(register _WINDOW *par, register _WINDOW *child,
	register _WINDOW *to, register RECT *r, register _REGION *rgn)
{
	if (to) to = to->lower;
    while (child != to) {
    	if _IS_DRAWABLE(child, *r) _update_windows_ex(child, r, rgn);
    	child = child->lower;
    }
    if (par) {
    	_calc_back_region(par);
        _invalidate_window_ex(par, rgn);
    }
}


/* redraws only invalidated windows */
static void _redraw_invalidated(_WINDOW *wnd, RECT *rect)
{
	RECT redraw_rect;

    if (!screen) return;
    if (!_IS_DRAWABLE(wnd, *rect)) return;
    INTERSECT_RECTS(_WND->clip, *rect, redraw_rect);
    scare_mouse();
    if (wnd->invalid_region.count) _DO_PROC(wnd, WE_PAINT, &redraw_rect);
    _TRAVERSE_WINDOWS(wnd, _redraw_invalidated(child, rect));
    unscare_mouse();
}


/* redraws changes at the background windows */
static void _redraw_bg(register _WINDOW *par, register _WINDOW *child,
	register _WINDOW *to, register RECT *r)
{
	scare_mouse();
	if (to) to = to->lower;
    while (child != to) {
    	if _IS_DRAWABLE(child, *r) _redraw_invalidated(child, r);
    	child = child->lower;
    }
    if (par) RedrawWindowRect((WINDOW)par, r, FALSE);
	unscare_mouse();
}


/* paint loop with clipping */
static BOOL _paint_loop_clipped(register _WINDOW *wnd, register RECT *rct,
	register BOOL overdraw_children)
{
	RECT t;

	/* check params */
	if (!wnd) return FALSE;
    
    /* get next area to clip */
	while (_paint_count) {
    	if RECTS_OVERLAP(*_paint_rect, _paint_clip) {
        	INTERSECT_RECTS(*_paint_rect, _paint_clip, t);
            set_clip(screen, t.left, t.top, t.right, t.bottom);
            _paint_rect++;
            _paint_count--;
            return TRUE;
        }
        _paint_rect++;
        _paint_count--;
    }

    /* end paint */
    unscare_mouse();
    _paint = FALSE;
    screen->cl = _saved_cl;
    screen->ct = _saved_ct;
    screen->cr = _saved_cr;
    screen->cb = _saved_cb;
    _paint_loop_proc = _paint_loop_init;
    wnd->invalid_region.count = 0;
    return FALSE;
}

    
/* paint loop without clipping */
static BOOL _paint_loop(register _WINDOW *wnd, register RECT *rct,
	register BOOL overdraw_children)
{
	/* check params */
	if (!wnd) return FALSE;
    
    /* get next area to clip */
	while (_paint_count) {
        set_clip(screen, _paint_rect->left, _paint_rect->top,
        	_paint_rect->right, _paint_rect->bottom);
        _paint_rect++;
        _paint_count--;
        return TRUE;
    }

    /* end paint */
    unscare_mouse();
    _paint = FALSE;
    screen->cl = _saved_cl;
    screen->ct = _saved_ct;
    screen->cr = _saved_cr;
    screen->cb = _saved_cb;
    _paint_loop_proc = _paint_loop_init;
    wnd->invalid_region.count = 0;
    return FALSE;
}

    
/* paint loop initializer */
static BOOL _paint_loop_init(register _WINDOW *wnd, register RECT *rct,
	register BOOL overdraw_children)
{
    if (!screen) return FALSE;
    
	/* check params */
	if (!wnd) return FALSE;

    /* window can not be drawn */
    if (!wnd->drawable) return FALSE;

    /* initialize paint loop if needed */
    if (!_paint) {
        /* get region to paint */
        if (wnd->invalid_region.count) {
        	_paint_rect = wnd->invalid_region.rects;
            _paint_count = wnd->invalid_region.count;
        }
        else if (!overdraw_children) {
        	_paint_rect = wnd->back_region.rects;
            _paint_count = wnd->back_region.count;
        }
        else {
        	_paint_rect = wnd->fore_region.rects;
            _paint_count = wnd->fore_region.count;
        }
        if (!_paint_count) return FALSE;

    	/* find if window overlaps with passed clipping */
    	if (rct) {
        	if (!RECTS_OVERLAP(wnd->clip, *rct)) return FALSE;
            INTERSECT_RECTS(wnd->clip, *rct, _paint_clip);
            _paint_loop_proc = _paint_loop_clipped;
        }
        else _paint_loop_proc = _paint_loop;

        /* begin paint */
    	_saved_cl = screen->cl;
    	_saved_ct = screen->ct;
    	_saved_cr = screen->cr;
    	_saved_cb = screen->cb;
        _paint = TRUE;
        scare_mouse();
    }

    /* start paint loop */
	return _paint_loop_proc(wnd, rct, overdraw_children);
}


/* returns child window at given z */
static INLINE _WINDOW *_child_at_z(register _WINDOW *wnd, register int z)
{
	register int c = 0;

	_TRAVERSE_WINDOWS(wnd,
    	if (c == z) return child;
        c++;
    );
    return NULL;
}


/* inserts a window in another window */
static void _insert_window(register _WINDOW *par, register _WINDOW *wnd,
	register int z)
{
	_WINDOW *t;

	/* parent contains one or more children */
	if (par->count) {
    	/* top position */
        if (z == 0) {
            wnd->lower = par->top;
            par->top->higher = wnd;
            par->top = wnd;
        }
        /* else bottom position */
        else if ((z < 0) || (z >= (int)par->count)) {
        	wnd->higher = par->bottom;
            par->bottom->lower = wnd;
            par->bottom = wnd;
        }
        /* else middle position */
        else {
        	t = _child_at_z(par, z);
            wnd->lower = t;
            wnd->higher = t->higher;
            t->higher->lower = wnd;
            t->higher = wnd;
        }
    }
    /* else initial child */
    else {
    	par->bottom = wnd;
        par->top = wnd;
    }

    wnd->parent = par;
    par->count++;
}


/* removes a window from its parent */
static void _remove_window(register _WINDOW *wnd)
{
	register _WINDOW *par = wnd->parent;

    if (wnd->lower) wnd->lower->higher = wnd->higher;
    else par->bottom = wnd->higher;
    if (wnd->higher) wnd->higher->lower = wnd->lower;
    else par->top = wnd->lower;
    par->count--;
    wnd->parent = NULL;
    wnd->lower = NULL;
    wnd->higher = NULL;
}


/* updates the screen after a window geometry change */
static void _update_screen(register _WINDOW *wnd)
{
    RECT prev_clipping, update_rect;
	register BOOL prev_drawable;
    register _WINDOW *par;

	/* check if window is drawn */
	if (!wnd->drawn) return;

    /* save current clipping */
    prev_drawable = wnd->drawable;
    prev_clipping = wnd->clip;

    /* re-render */
    _render_windows(wnd);

    /* update windows in background if window is a child */
    par = wnd->parent;
    if (par) {
    	/* window is and was drawable */
        if (wnd->drawable && prev_drawable) {
        	UNITE_RECTS(wnd->clip, prev_clipping, update_rect);
            _calc_bg(par, wnd->lower, NULL, &update_rect);
            _invalidate_bg(par, wnd->lower, NULL, &prev_clipping);
            _redraw_bg(par, wnd->lower, NULL, &prev_clipping);
        }
        /* else window was drawable but now isn't */
        else if (!wnd->drawable && prev_drawable) {
            _update_bg(par, wnd->lower, NULL, &prev_clipping);
            _redraw_bg(par, wnd->lower, NULL, &prev_clipping);
        }
        /* else window is drawable; previously it was not */
        else if (wnd->drawable && !prev_drawable) {
            _calc_bg(par, wnd->lower, NULL, &wnd->clip);
        }
    }
    RedrawWindowRect((WINDOW)wnd, NULL, TRUE);
}


/* destroys windows recursively */
static void _destroy_windows(register _WINDOW *wnd)
{
	_TRAVERSE_WINDOWS(wnd, _destroy_windows(child));
    if (wnd->fore_region.rects) free(wnd->fore_region.rects);
    if (wnd->back_region.rects) free(wnd->back_region.rects);
    if (wnd->invalid_region.rects) free(wnd->invalid_region.rects);
    free(wnd);
}


/* deletes timers of the given window and its descentants */
static INLINE void _delete_timers(register WINDOW wnd)
{
	_lock_events();
    _TRAVERSE_TIMERS(
    	if (tm->wnd) {
        	if ((tm->wnd == wnd) || IsAncestorWindow(tm->wnd, wnd))
            	tm->wnd = NULL;
        }
    );
    _unlock_events();
}


/* deletes events of the given window and its descentants */
static INLINE void _delete_events(register WINDOW wnd)
{
	register int i, max, c;
    register EVENT *e;

    max = _ready_queue.count;
    if (!max) return;
    c = _ready_queue.avail;
    e = &_ready_queue.events[c];
    for(i = 0; i < max; i++) {
    	if (e->mouse.wnd) {
        	if ((e->mouse.wnd == wnd) ||
                IsAncestorWindow(e->mouse.wnd, wnd)) {
            	e->type = WE_NONE;
                e->mouse.wnd = NULL;
                e->mouse.dst = NULL;
            }
        }
    	c++;
        e++;
        if (c == _MAX_EVENTS) {
        	c = 0;
    		e = &_ready_queue.events[0];
        }
    }
}


/* clears all the timers */
static INLINE void _clear_timers()
{
	_lock_events();
    memset(&_timers, 0, sizeof(_timers));
    _unlock_events();
}


/* clears an event queue */
static INLINE void _clear_queue(register _QUEUE *queue)
{
	queue->free = 0;
    queue->avail = 0;
    queue->count = 0;
}


/* puts an event in the ready queue */
static void _put_ready_event(register EVENT *e, register int type,
	register WINDOW wnd)
{
	EVENT *t;

    /* check params */
    if (!wnd) return;

    /* allocate an event from the ready queue */
    t = _alloc_event(&_ready_queue);
    if (!t) return;

    /* set up the event */
    *t = *e;
    t->type = type;
    t->mouse.wnd = wnd;

    /* set up the destination window for the event; if there is a grab
       window, the destination is the grab window; else
       the destination is the window itself */
    if (!_grab) t->mouse.dst = wnd; else t->mouse.dst = _grab;
}


/* processes a raw event and converts it to an event for a window */
static INLINE void _process_raw_event(register EVENT *e)
{
	WINDOW prev_mouse;

	/* must have root window */
    if (!_root) return;

    /* process event */
    switch (e->type) {
    	/* mouse move */
        case WE_MOUSEMOVE:
        	prev_mouse = _mouse;
            _mouse = WindowFromPoint(_root, e->mouse.x, e->mouse.y);
            if (prev_mouse == _mouse) _put_ready_event(e, e->type, _mouse);
            else {
                 _put_ready_event(e, WE_MOUSELEAVE, prev_mouse);
                 _put_ready_event(e, WE_MOUSEENTER, _mouse);
            }
        	return;

        /* static events */
        case WE_LBUTTONDOWN:
        case WE_RBUTTONDOWN:
        case WE_MBUTTONDOWN:
        case WE_LBUTTONUP:
        case WE_RBUTTONUP:
        case WE_MBUTTONUP:
        	if (!_mouse) _mouse = WindowFromPoint(_root, e->mouse.x, e->mouse.y);
            _put_ready_event(e, e->type, _mouse);
        	return;

        /* key events */
        case WE_KEYDOWN:
        case WE_KEYUP:
            _put_ready_event(e, e->type, _focus);
        	return;

        /* timer event */
        case WE_TIMER:
            _put_ready_event(e, e->type, _timers[e->timer.slot].wnd);
        	return;
    }
}


/* creates a bitmap copy at the given color depth */
static BITMAP *_copy_bitmap(register BITMAP *bmp, register int depth)
{
	BITMAP *result;

    /* create a new bitmap */
	result = create_bitmap_ex(depth, bmp->w, bmp->h);
    if (!result) return NULL;

    /* copy bitmap */
	blit(bmp, result, 0, 0, 0, 0, bmp->w, bmp->h);
    return result;
}


/* checks if the current bitmap of the cursor has been created
   and it is appropriate for the current video mode; if so,
   it returns TRUE; */
static INLINE BOOL _is_proper_cursor(register _CURSOR *cur)
{
    return cur->current &&
    	bitmap_color_depth(cur->current) ==
    		bitmap_color_depth(screen);
}


/* renders the current bitmap cursor to the current color depth,
   only if needed (i.e. only if the cursor has not been rendered before
   or the color depth of the screen is different than that of
   the current bitmap of the cursor) */
static void _render_cursor(register _CURSOR *cur)
{
	if (_is_proper_cursor(cur)) return;
   	if (cur->current) destroy_bitmap(cur->current);
    cur->current = _copy_bitmap(cur->bitmap, bitmap_color_depth(screen));
}


/* processes a ready event for various FXs */
static void _process_ready_event(register EVENT *e)
{
	register WINDOW wnd;

	switch (e->type) {
    	/* mouse enter */
        case WE_MOUSEENTER:
        	wnd = e->mouse.wnd;
        	if (wnd && _WND->cursor) SetCursor(_WND->cursor);
        	return;
    }
}


/*************************************************************************
	FUNCTIONS
 *************************************************************************/


/* this must be public, but you should use the BEGIN_PAINT / END_PAINT
   macros */
BOOL _do_paint_loop(WINDOW wnd, RECT *rct, BOOL overdraw_children)
{
    return _paint_loop_proc(_WND, rct, overdraw_children);
}


/* returns the parent window */
WINDOW GetParentWindow(WINDOW wnd)
{
	if (wnd) return (WINDOW)_WND->parent;
    return NULL;
}


/* returns the lower (in z-order) sibling window */
WINDOW GetLowerWindow(WINDOW wnd)
{
	if (wnd) return (WINDOW)_WND->lower;
    return NULL;
}


/* returns the higher (in z-order) sibling window */
WINDOW GetHigherWindow(WINDOW wnd)
{
	if (wnd) return (WINDOW)_WND->higher;
    return NULL;
}


/* returns the bottom (in z-order) child window */
WINDOW GetBottomWindow(WINDOW wnd)
{
	if (wnd) return (WINDOW)_WND->bottom;
    return NULL;
}


/* returns the top (in z-order) child window */
WINDOW GetTopWindow(WINDOW wnd)
{
	if (wnd) return (WINDOW)_WND->top;
    return NULL;
}


/* returns the number of children windows */
unsigned GetWindowCount(WINDOW wnd)
{
	if (wnd) return _WND->count;
    return 0;
}


/* returns the x coordinate of the window */
short GetWindowX(WINDOW wnd)
{
	if (wnd) return _WND->x;
    return 0;
}


/* returns the y coordinate of the window */
short GetWindowY(WINDOW wnd)
{
	if (wnd) return _WND->y;
    return 0;
}


/* returns the width of the window */
USHORT GetWindowWidth(WINDOW wnd)
{
	if (wnd) return _WND->width;
    return 0;
}


/* returns the height of the window */
USHORT GetWindowHeight(WINDOW wnd)
{
	if (wnd) return _WND->height;
    return 0;
}


/* returns the viewport width of the window */
USHORT GetWindowViewportWidth(WINDOW wnd)
{
	if (wnd) return _WND->width - _WND->frame.left - _WND->frame.right;
    return 0;
}


/* returns the viewport height of the window */
USHORT GetWindowViewportHeight(WINDOW wnd)
{
	if (wnd) return _WND->height - _WND->frame.top - _WND->frame.bottom;
    return 0;
}


/* returns the z-order of a window */
int GetWindowZOrder(WINDOW wnd)
{
	register int z = -1;

	while (wnd) {
    	z++;
        wnd = (WINDOW)_WND->higher;
    }
    return z;
}


/* returns the frame of the window */
void GetWindowFrame(WINDOW wnd, RECT *frame)
{
	if (wnd && frame) *frame = _WND->frame;
}


/* returns the screen position of the window */
void GetWindowRect(WINDOW wnd, RECT *rct)
{
	if (wnd && rct) *rct = _WND->pos;
}


/* returns the window cursor */
CURSOR GetWindowCursor(WINDOW wnd)
{
	if (wnd) return _WND->cursor;
    return NULL;
}


/* returns the window proc */
WINDOW_PROC *GetWindowProc(WINDOW wnd)
{
	if (wnd) return _WND->proc;
    return NULL;
}


/* returns the window data */
void *GetWindowData(WINDOW wnd)
{
	if (wnd) return _WND->data;
    return NULL;
}


/* returns the window flags */
unsigned GetWindowFlags(WINDOW wnd)
{
	if (wnd) return _WND->flags;
    return 0;
}


/* returns TRUE if window B is ancestor of window A */
BOOL IsAncestorWindow(WINDOW a, WINDOW b)
{
	while (a) {
    	a = (WINDOW)_MK_WND(a)->parent;
        if (a == b) return TRUE;
    }
    return FALSE;
}


/* redraws part of a window */
BOOL RedrawWindowRect(WINDOW wnd, RECT *rect, BOOL redraw_children)
{
	RECT redraw_rect;

	if (_paint) return FALSE;
    if (!screen) return FALSE;

	/* check params */
    if (!_WND) return FALSE;
    if (!_WND->drawable) return FALSE;

    /* check if window overlaps with passed area */
    if (rect) {
    	if (!RECTS_OVERLAP(_WND->clip, *rect)) return FALSE;
        INTERSECT_RECTS(_WND->clip, *rect, redraw_rect);
        rect = &redraw_rect;
    }

    /* begin paint */
    scare_mouse();
    
    /* paint window */
    _DO_PROC(wnd, WE_PAINT, rect);

    /* paint children if requested */
    if ((redraw_children) || (!(_WND->flags & WF_CLIPCHILDREN))) {
        _TRAVERSE_WINDOWS(wnd,
        	RedrawWindowRect((WINDOW)child, rect, redraw_children);
        );
    }
    
    /* end paint */
    unscare_mouse();
    return TRUE;
}


/* redraws the whole window */
BOOL RedrawWindow(WINDOW wnd, BOOL redraw_children)
{
	if (_paint) return FALSE;
    if (!_WND) return FALSE;
    return RedrawWindowRect(wnd, NULL, redraw_children);
}


/* calls a window's proc with the given message and message data */
int SendMessage(WINDOW wnd, int msg, void *msg_data)
{
	return _DO_PROC(wnd, msg, msg_data);
}


/* creates a window */
WINDOW CreateWindow(WINDOW_PROC *proc, WINDOW parent,
	short x, short y, USHORT width, USHORT height,
    unsigned flags, void *data)
{
	register _WINDOW *wnd;

    _install();

	if (_paint) return FALSE;
    
	/* check params */
    if (!proc) return NULL;

    /* allocate a window struct */
    wnd = calloc(1, sizeof(_WINDOW));
    if (!wnd) return NULL;

    /* set up window from passed parameters */
    wnd->proc = proc;
    wnd->x = x;
    wnd->y = y;
    wnd->width = width;
    wnd->height = height;
    wnd->flags = flags;
    wnd->data = data;

    /* insert into parent */
    MapWindow(parent, (WINDOW)wnd, 0);

    return (WINDOW)wnd;
}


/* destroys a window and its children; returns FALSE if given window
   is NULL */
BOOL DestroyWindow(WINDOW wnd)
{
	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;

    /* remove from parent or hide root window */
    if (_WND->parent) UnmapWindow(wnd);
    else if (wnd == _root) HideWindow(wnd);

    /* destroy windows recursively */
    _destroy_windows(_WND);
    return TRUE;
}


/* inserts a window in another window; returns NULL if window is
   already a child of another window */
BOOL MapWindow(WINDOW parent, WINDOW wnd, int z_order)
{
	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;
    if (!parent) return FALSE;
    if (_WND->parent) return FALSE;
    if (wnd == parent) return FALSE;
    if (wnd == _root) return FALSE;
    if (IsAncestorWindow(parent, wnd)) return FALSE;

    /* insert window in parent window list */
    _insert_window(_PARENT, _WND, z_order);

    /* update screen */
    if (!_PARENT->drawn) return TRUE;
    _render_windows(_WND);
    if (!_WND->drawable) return TRUE;
    _calc_bg(_PARENT, _WND->lower, NULL, &_WND->clip);
    RedrawWindowRect(wnd, NULL, TRUE);
    return TRUE;
}


/* removes a window from its parent */
BOOL UnmapWindow(WINDOW wnd)
{
	register _WINDOW *par, *lo;

	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;
    par = _WND->parent;
    if (!par) return FALSE;

    /* remove from parent window list */
    lo = _WND->lower;
    _remove_window(_WND);

    /* update screen */
    if (!_WND->drawn) return TRUE;

    /* reset internal stuff that depend on the given window */
    if (_mouse) {
    	if ((wnd == _mouse) || (IsAncestorWindow(_mouse, wnd)))
        	_mouse = NULL;
    }
    if (_focus) {
    	if ((wnd == _focus) || (IsAncestorWindow(_focus, wnd)))
        	_focus = NULL;
    }
    if (_grab) {
    	if ((wnd == _grab) || (IsAncestorWindow(_grab, wnd)))
        	_grab = NULL;
    }
    _delete_timers(wnd);
    _delete_events(wnd);

    /* update background */
    if (_WND->drawable) {
	    _unrender_windows(_WND);
    	_update_bg(par, lo, NULL, &_WND->clip);
        _redraw_bg(par, lo, NULL, &_WND->clip);
        return TRUE;
    }

    /* only unrender given window (if window not drawable) */
    _unrender_windows(_WND);
    return TRUE;
}


/* resizes a window; returns FALSE if given position and size
   is the same with the current one */
BOOL ResizeWindow(WINDOW wnd,
	short x, short y, USHORT width, USHORT height,
    BOOL repaint)
{
	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;

    /* check if there is a geometry change */
    if (_WND->x == x &&
    	_WND->y == y &&
        _WND->width == width &&
        _WND->height == height)
        return FALSE;

    /* set new geometry */
    _WND->x = x;
    _WND->y = y;
    _WND->width = width;
    _WND->height = height;

    /* update screen */
    if (repaint) _update_screen(_WND);
    return TRUE;
}


/* sets the frame of the given window */
BOOL SetWindowFrame(WINDOW wnd, RECT *frame)
{
	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;
    if (!frame) return FALSE;

    /* set frame */
    _WND->frame = *frame;

    /* update screen */
    if (!_WND->drawable) return TRUE;
    _render_windows(_WND);
    RedrawWindowRect(wnd, NULL, TRUE);
    return TRUE;
}


/* sets the z-order of a window; returns FALSE if given z-order
   is the same as the current z-order */
BOOL SetWindowZOrder(WINDOW wnd, int z_order)
{
    _REGION prev_region, update_region;
	register _WINDOW *par, *lo, *hi;
	register int prev_z;

	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;

    /* check if window has a parent */
    par = _WND->parent;
    if (!par) return FALSE;

    /* check for different z */
    if ((z_order < 0) || (z_order >= (int)par->count)) z_order = par->count - 1;
    prev_z = GetWindowZOrder(wnd);
    if (z_order == prev_z) return FALSE;

    /* remove from parent and re-insert to the new z-order */
    lo = _WND->lower;
    hi = _WND->higher;
	_remove_window(_WND);
    _insert_window(par, _WND, z_order);

    /* update screen only if window is drawable */
    if (!_WND->drawable) return TRUE;

    /* store current region and calculate the new one */
	memset(&prev_region, 0, sizeof(_REGION));
	memset(&update_region, 0, sizeof(_REGION));
    _copy_region(&_WND->fore_region, &prev_region);
 	_recalc_regions(_WND, &_WND->clip);
    
    /* raised */
    if (z_order < prev_z) {
    	/* calculate update area */
        _subtract_regions(&_WND->fore_region, &prev_region, &update_region);

        /* update windows according to update region */
        _invalidate_windows_ex(_WND, &_WND->clip, &update_region);
        
    	/* recalculate sibling windows in background */
    	_calc_bg(NULL, _WND->lower, hi, &_WND->clip);

        /* redraw changes */
        RedrawWindowRect(wnd, NULL, TRUE);
    }
    /* else lowered */
    else {
    	/* calculate update region */
        _subtract_regions(&prev_region, &_WND->fore_region, &update_region);

        /* update background */
   		_update_bg_ex(NULL, lo, _WND->higher, &_WND->clip, &update_region);

        /* redraw changes in background */
        _redraw_bg(NULL, lo, _WND->higher, &_WND->clip);
    }

    /* free temp regions */
    if (prev_region.rects) free(prev_region.rects);
    if (update_region.rects) free(update_region.rects);
    return TRUE;
}


/* sets a window's cursor; if the window has a cursor, then
   the cursor is set automatically when the mouse pointer is
   over the window */
BOOL SetWindowCursor(WINDOW wnd, CURSOR cur)
{
	if (_paint) return FALSE;
	if (!wnd) return FALSE;
    if (!cur) return FALSE;
    _WND->cursor = cur;
    return TRUE;
}


/* sets a window's procedure; it does not redraw the window */
BOOL SetWindowProc(WINDOW wnd, WINDOW_PROC *proc)
{
	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;
    if (!proc) return FALSE;

    /* set proc */
    _WND->proc = proc;
    return TRUE;
}


/* sets a window's data pointer; it does not redraw the window  */
BOOL SetWindowData(WINDOW wnd, void *data)
{
	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;

    /* set data */
    _WND->data = data;
    return TRUE;
}


/* shows a window; returns FALSE if window is already shown */
BOOL ShowWindow(WINDOW wnd)
{
	register _WINDOW *par;

	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;

    /* showing a non-child window makes a root window visible */
    par = _WND->parent;
    
    /* show child */
    if (par) {
    	if (_WND->flags & WF_VISIBLE) return FALSE;
        _WND->flags |= WF_VISIBLE;
        if (!_WND->drawn) return TRUE;
        _render_windows(_WND);
        if (!_WND->drawable) return TRUE;
        _calc_bg(par, _WND->lower, NULL, &_WND->clip);
	    RedrawWindowRect(wnd, NULL, TRUE);
    	return TRUE;
    }
    
    /* show root */
    if (wnd == _root) return FALSE;    
	HideWindow(_root);
    _root = wnd;
    _render_windows(_WND);
    RedrawWindowRect(wnd, NULL, TRUE);
	return TRUE;
}


/* hides a window; returns FALSE if window is already hidden */
BOOL HideWindow(WINDOW wnd)
{
	register _WINDOW *par;

	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;

    par = _WND->parent;

    /* hide child */
    if (par) {
	    if (!(_WND->flags & WF_VISIBLE)) return FALSE;
    	_WND->flags &= ~WF_VISIBLE;
        if (!_WND->drawn) return TRUE;
        if (_WND->drawable) {
	        _render_windows(_WND);
            _update_bg(par, _WND->lower, NULL, &_WND->clip);
            _redraw_bg(par, _WND->lower, NULL, &_WND->clip);
        }
        else {
	        _render_windows(_WND);
        }
    	return TRUE;
    }

    /* hide root */
    if (wnd == _root) {
        _unrender_windows(_WND);
    	_root = NULL;
        _mouse = NULL;
        _focus = NULL;
        _grab = NULL;
        _clear_timers();
        _clear_queue(&_ready_queue);
    	return TRUE;
    }

    /* hide an non-child window */
    if (!(_WND->flags & WF_VISIBLE)) return FALSE;
    _WND->flags &= ~WF_VISIBLE;
    return TRUE;
}


/* enables a window in order to accept events; returns FALSE if window
   is already enabled */
BOOL EnableWindow(WINDOW wnd)
{
	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;

    /* if not enabled, enable and redraw */
  	if (_WND->flags & WF_ENABLED) return FALSE;
    _WND->flags |= WF_ENABLED;
    RedrawWindowRect(wnd, NULL, FALSE);
    return TRUE;
}


/* disables a window; returns FALSE if window is already disabled */
BOOL DisableWindow(WINDOW wnd)
{
	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;

    /* if not disabled, disable and redraw */
  	if (!(_WND->flags & WF_ENABLED)) return FALSE;
    _WND->flags &= ~WF_ENABLED;
    RedrawWindowRect(wnd, NULL, FALSE);
    return TRUE;
}


/* sets the input focus to the given window; returns FALSE if
   the given window already has the focus */
BOOL SetFocus(WINDOW wnd)
{
	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;
    if (wnd == _focus) return FALSE;
    if (!_WND->drawn) return FALSE;
  	if (!(_WND->flags & WF_ENABLED)) return FALSE;

    /* remove the focus from the previous window and set the focus
       to the new one */
    RemoveFocus(_focus);
    _focus = wnd;
    RedrawWindowRect(wnd, NULL, FALSE);
    return TRUE;
}


/* removes the focus from the given window;
   returns FALSE if no window had the input focus */
BOOL RemoveFocus(WINDOW wnd)
{
	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;
    if (wnd != _focus) return FALSE;

    /* remove the focus */
    _focus = NULL;
    RedrawWindowRect(wnd, NULL, FALSE);
    return TRUE;
}


/* the given window grabs mouse and keyboard events; all events
   are redirected to it until the window releases events; returns FALSE
   if another window has grabbed events */
BOOL GrabEvents(WINDOW wnd)
{
	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;
    if (_grab) return FALSE;
    if (!_WND->drawn) return FALSE;
  	if (!(_WND->flags & WF_ENABLED)) return FALSE;

    /* set the event window */
    _grab = wnd;
    return TRUE;
}


/* releases events from the given window; if no window
   had grabbed events, then it returns FALSE */
BOOL ReleaseEvents(WINDOW wnd)
{
	if (_paint) return FALSE;
    
	/* check params */
    if (!wnd) return FALSE;
    if (wnd != _grab) return FALSE;

    /* remove the event window */
    _grab = NULL;
    return TRUE;
}


/* returns the root window */
WINDOW GetRootWindow()
{
	return _root;
}


/* returns the window with the focus */
WINDOW GetFocusWindow()
{
	return _focus;
}


/* returns the window which has captured events */
WINDOW GetEventWindow()
{
	return _grab;
}


/* returns window under the given x, y coordinates starting
   from the given window; coordinates are relative to the screen */
WINDOW WindowFromPoint(WINDOW wnd, short x, short y)
{
	register WINDOW result;

	/* check params */
    if (!wnd) return FALSE;
	if (!POINT_IN_RECT(_WND->clip, x, y)) return FALSE;
	_TRAVERSE_WINDOWS(wnd,
    	result = WindowFromPoint((WINDOW)child, x, y);
        if (result) return result;
    );
    if (_point_in_region(&_WND->back_region, x, y)) return wnd;
    return NULL;
}


/* returns the next event from the event queue */
BOOL GetEvent(EVENT *event)
{
	register EVENT *t;

	if (_paint) return FALSE;

    /* check params */
	if (!event) return FALSE;

    /* get a raw event */
    event->type = WE_NONE;
    _lock_events();
    t = _free_event(&_raw_queue);
    if (t) *event = *t;
    _unlock_events();

    /* process a raw event */
    if (event->type != WE_NONE) _process_raw_event(event);

    /* get a ready event */
    event->type = WE_NONE;
    t = _free_event(&_ready_queue);
    if (t && (t->type != WE_NONE)) {
    	*event = *t;
        _process_ready_event(t);
        return TRUE;
    }

    /* no event found */
	return FALSE;
}


/* dispatches an event to the destination window of the event,
   only if the destination window is enabled */
int DispatchEvent(EVENT *event)
{
	if (event &&
    	event->mouse.dst &&
    	((_MK_WND(event->mouse.dst)->flags & WF_ENABLED)))
    	return _DO_PROC(event->mouse.dst, event->type, event);
    return 0;
}


/* sets a timer in the given window; returns timer slot or 0
   if no timer is available */
unsigned SetTimer(WINDOW wnd, unsigned timeout)
{
	register int res;

	if (_paint) return 0;

    /* check params */
    if (!wnd) return 0;
    if (!_WND->drawn) return 0;
  	if (!(_WND->flags & WF_ENABLED)) return FALSE;

    /* set timer */
    res = 0;
    _lock_events();
    _TRAVERSE_TIMERS(
    	if (!tm->wnd) {
        	tm->timer = 0;
            tm->timeout = timeout;
            tm->wnd = wnd;
            res = i;
            break;
        }
    );
    _unlock_events();
    
	return res;
}


/* releases a timer */
BOOL RemoveTimer(WINDOW wnd, unsigned timer_slot)
{
	register BOOL res;

	if (_paint) return FALSE;

    /* check params */
    if (!wnd) return FALSE;

    /* remove timer */
    res = FALSE;
    _lock_events();
    if ((timer_slot >= 1) &&
        (timer_slot < _MAX_EVENTS) &&
        (_timers[timer_slot].wnd == wnd)) {
        _timers[timer_slot].wnd = NULL;
        res = TRUE;
    }
    _unlock_events();
    return res;
}


/* returns the click timeout */
unsigned GetClickTimeout()
{
	return _click_timeout;
}


/* sets the click timeout */
void SetClickTimeout(unsigned milisecs)
{
	_click_timer = 0;
    _click_timeout = milisecs;
}


/* resets the number of clicks */
void ResetClicks()
{
	_click_timer = 0;
    _clicks = 0;
}


/* returns the mouse button values */
void GetMouseButtons(int *left, int *right, int *middle)
{
	if (left) *left = _buttons[1];
	if (right) *right = _buttons[2];
	if (middle) *middle = _buttons[3];
}


/* sets the mouse buttons */
void SetMouseButtons(int left, int right, int middle)
{
	_buttons[1] = left;
	_buttons[2] = right;
	_buttons[3] = middle;
}


/* creates a cursor from the given bitmap */
CURSOR CreateCursor(BITMAP *bmp, int x, int y)
{
	_CURSOR *cur;

	if (_paint) return NULL;

    /* check params */
    if (!bmp) return NULL;

    /* allocate a cursor */
    cur = calloc(1, sizeof(_CURSOR));
    if (!cur) return NULL;

    /* copy the given bitmap to the cursor */
    cur->bitmap = _copy_bitmap(bmp, bitmap_color_depth(bmp));

    /* failed to create a bitmap copy */
    if (!cur->bitmap) {
    	free(cur);
        return NULL;
    }

    /* set up the cursor hot spot */
    cur->x = x;
    cur->y = y;

    /* return the created cursor */
    return (CURSOR)cur;
}


/* destroy a cursor */
BOOL DestroyCursor(CURSOR cur)
{
	if (_paint) return FALSE;

    /* check params */
    if (!cur) return FALSE;

    /* handle the current cursor */
    if (cur == _curr_cursor) _curr_cursor = NULL;

    /* destroy the bitmaps of the cursor and the cursor */
	destroy_bitmap(_CUR->bitmap);
    if (_CUR->current) destroy_bitmap(_CUR->current);
    free(cur);
    return TRUE;
}


/* returns the current cursor */
CURSOR GetCursor()
{
	return _curr_cursor;
}


/* sets the current cursor; pass NULL to hide the mouse */
BOOL SetCursor(CURSOR cur)
{
	if (_paint) return FALSE;
    if (!screen) return FALSE;

    /* check cursor */
    if ((cur == _curr_cursor) && (!cur || _is_proper_cursor(_CUR)))
    	return TRUE;

    /* hide mouse in order to change cursor shape */
  	show_mouse(NULL);
    
    /* cursor given; set mouse sprite and mouse sprite focus */
    if (cur) {
    	_render_cursor(_CUR);
        if (_CUR->current) {
	    	set_mouse_sprite(_CUR->current);
    	    set_mouse_sprite_focus(_CUR->x, _CUR->y);
	        show_mouse(screen);
        }
        else return FALSE;
    }

    /* save new cursor */
    _curr_cursor = cur;
    return TRUE;
}


