/*
     _______                     ___                          ________
    /       \         /\        |   |\             /\        |        \
   /         >       /  \       |   ||            /  \       |         \
  /   ______/ >     /    \      |   ||           /    \      |    __    \
 <   <_______/     /      \     |   ||          /      \     |   |\_\    \
  \        \      /   /\   \    |   ||         /   /\   \    |   ||  \    \
   \        \    |   /_L\   |   |   ||        |   /_L\   |   |   ||   >   |\
    \_____   \   |          |\  |   ||        |          |\  |   ||  /    /|
   __L____>   >  |          ||  |   |L____    |          ||  |   |L_/    / /
  /          / > |   ____   ||  |         |\  |   ____   ||  |          / /
 <          / /  |   |\_|   ||  |         ||  |   |\_|   ||  |         / /
  \________/ /   |___|| |___||  |_________||  |___|| |___||  |________/ /
   \________/     \___\  \___\   \_________\   \___\  \___\   \_______\/


                an Addon Package for Allegro by Sven Sandberg


This file contains functions for the structs TRECT and TPOINT. Most of them
are connected to rectangles in some way.

*/
#ifndef s_rect_c
#define s_rect_c

#include "s_rect.h"
#include "s_defs.h"
#include "s_math.h"

#include <math.h>

#include <allegro.h>



//These are returned by some functions indicating error.
TPOINT nopoint={max_slong, max_slong};
T3DPOINT no3dpoint={max_slong, max_slong, max_slong};
TRECT norect={max_slong, max_slong, min_slong, min_slong};
TTWOPOINT notwopoint={max_slong, max_slong, max_slong, max_slong};
TRANGE norange={max_slong, min_slong};

TPOINT zeropoint={0, 0};
T3DPOINT zero3dpoint={0, 0};
TRECT zerorect={0, 0, 0, 0};
TTWOPOINT zerotwopoint={0, 0, 0, 0};
TRANGE zerorange={0,0};



/**************************
****                   ****
**** set_clip_fromrect ****
****                   ****
***************************
Sets clipping rectangle for a bitmap to a TRECT.
*/
void set_clip_fromrect(BITMAP *bmp,TRECT r)
{
bmp->cl=r.x;
bmp->cr=r.x+r.w;
bmp->ct=r.y;
bmp->cb=r.y+r.h;
}
/************************
****                 ****
**** get_clip_asrect ****
****                 ****
*************************
Returns clipping rectangle for a bitmap as a TRECT.
*/
TRECT get_clip_asrect(BITMAP *bmp)
{
TRECT ret={bmp->cl, bmp->ct, bmp->cr-bmp->cl, bmp->cb-bmp->ct};
return ret;
}
/*****************
****          ****
**** get_size ****
****          ****
******************
Returns a rect containing the size of a bitmap. x and y is always 0.
*/
TRECT get_size(BITMAP *bmp)
{
TRECT ret={0,0,bmp->w,bmp->h};
return ret;
}
/*******************
****            ****
**** reset_clip ****
****            ****
********************
Sets clipping for the bitmap to its width and height.
*/
void reset_clip(BITMAP *bmp)
{
bmp->cr=bmp->w;
bmp->cb=bmp->h;
}



/******************
****           ****
**** intersect ****
****           ****
*******************
Returns `TRUE' if r1 intersects r2, or `FALSE' if it does not.
NOT TESTED!
*/
int intersect(TRECT r1,TRECT r2)
{
if(r1.x<r2.x){
	if((r1.x+r1.w)>r2.x){
		if(r1.y<r2.y){
			if((r1.y+r1.h)>r2.y)
				return TRUE;
			else
				return FALSE;
		}else if(r1.y<(r2.y+r2.h))
			return TRUE;
		else
			return FALSE;
	}else
		return FALSE;
}else if(r1.x<(r2.x+r2.w)){
	if(r1.y<r2.y){
		if((r1.y+r1.h)>r2.y)
			return TRUE;
		else
			return FALSE;
	}else if(r1.y<(r2.y+r2.h))
		return TRUE;
	else
		return FALSE;
}else
	return FALSE;
}
/********************
****             ****
**** pointinrect ****
****             ****
*********************
Returns `TRUE' if the point is inside the rect.
NOT TESTED!
*/
inline int pointinrect(TPOINT p,TRECT r)
{
return (p.x >= r.x) && (p.x < (r.x+r.w)) &&
		 (p.y >= r.y) && (p.y < (r.y+r.h));
}

/*****************
****          ****
**** midpoint ****
****          ****
******************
Returns a point between two points.
*/
TPOINT midpoint(TPOINT p1,TPOINT p2)
{
p1.x = (p1.x+p2.x)/2;
p1.y = (p1.y+p2.y)/2;
return p1;
}

/******************
****           ****
**** bothfitin ****
****           ****
*******************
Returns a TRECT that is big enough to contain both the parameter rectangles.
NOT TESTED!
*/
TRECT bothfitin(TRECT r1,TRECT r2)
{
//r1 is used to keep the result in.
//Clipping x and w.
if(r2.x<r1.x){
	if(r2.x+r2.w>r1.x+r1.w)
		r1.w=r2.w;
	else
		r1.w+=r1.x-r2.x;
	r1.x=r2.x;
}else
	if(r2.x+r2.w>r1.x+r1.w)
		r1.w=r2.x+r2.w-r1.x;
//Clipping y and h.
if(r2.y<r1.y){
	if(r2.y+r2.h>r1.y+r1.h)
		r1.h=r2.h;
	else
		r1.h+=r1.y-r2.y;
	r1.y=r2.y;
}else
	if(r2.y+r2.h>r1.y+r1.h)
		r1.h=r2.y+r2.h-r1.y;
return r1;
}
/******************
****           ****
**** fitinboth ****
****           ****
*******************
Returns a TRECT which is inside both the parameter rectangles. Obviously,
this is only possible if they intersect. If they don't, the function will
return a 0,0,0,0-rect.
NOT TESTED!
*/
TRECT fitinboth(TRECT r1,TRECT r2)
{
//r1 is used to keep the result in.
//Clipping x and w.
if(r1.x<r2.x){
	if(r1.x+r1.w>r2.x+r2.w){   //Start before && end after.
		r1.x=r2.x;
		r1.w=r2.w;
	}else if(r1.x+r1.w>r2.x){  //Start before && end between.
		r1.w+=r1.x-r2.x;
		r1.x=r2.x;
	}else                      //Start before && end before -> error.
		return norect;
}else if(r1.x<r2.x+r2.w){
	if(r1.x+r1.w>r2.x+r2.w)    //Start between && end after.
		r1.w=r2.x+r2.w-r1.x;
										//Start between && end between -> do nothing.
}else                         //Start after -> error.
	return norect;
//Clipping y and h.
if(r1.y<r2.y){
	if(r1.y+r1.h>r2.y+r2.h){   //Start before && end after.
		r1.y=r2.y;
		r1.h=r2.h;
	}else if(r1.y+r1.h>r2.y){  //Start before && end between.
		r1.h+=r1.y-r2.y;
		r1.y=r2.y;
	}else                      //Start before && end before -> error.
		return norect;
}else if(r1.y<r2.y+r2.h){
	if(r1.y+r1.h>r2.y+r2.h)    //Start between && end after.
		r1.h=r2.y+r2.h-r1.y;
										//Start between && end between -> do nothing.
}else                         //Start after -> error.
	return norect;
return r1;
}

TRECT resizerect(TRECT r,int bypixels)
{
r.x-=bypixels;
r.y-=bypixels;
r.w+=(bypixels*2);
r.h+=(bypixels*2);
return r;
}


//Moving one side of the rectangle.
inline TRECT moveleft(TRECT r,int x)
{
r.x+=x;
r.w-=x;
return r;
}
inline TRECT movetop(TRECT r,int y)
{
r.y+=y;
r.h-=y;
return r;
}
inline TRECT moveright(TRECT r,int x)
{
r.w+=x;
return r;
}
inline TRECT movebottom(TRECT r,int y)
{
r.h+=y;
return r;
}
//Moving one corner of the rectangle.
inline TRECT movetopleft(TRECT r,int x,int y)
{
r.x+=x;
r.w-=x;
r.y+=y;
r.h-=y;
return r;
}
inline TRECT movetopright(TRECT r,int x, int y)
{
r.w+=x;
r.y+=y;
r.h-=y;
return r;
}
inline TRECT movebottomleft(TRECT r,int x, int y)
{
r.x+=x;
r.w-=x;
r.y+=y;
return r;
}
inline TRECT movebottomright(TRECT r,int x, int y)
{
r.w+=x;
r.h+=y;
return r;
}
//Placing one side of the rectangle.
inline TRECT placeleft(TRECT r,int x)
{
r.w+=r.x-x;
r.x=x;
return r;
}
inline TRECT placetop(TRECT r,int y)
{
r.h+=r.y-y;
r.y=y;
return r;
}
inline TRECT placeright(TRECT r,int x)
{
r.w=x-r.x;
return r;
}
inline TRECT placebottom(TRECT r,int y)
{
r.h=y-r.y;
return r;
}
//Placing one corner of the rectangle.
inline TRECT placetopleft(TRECT r,int x,int y)
{
r.w+=r.x-x;
r.x=x;
r.h+=r.y-y;
r.y=y;
return r;
}
inline TRECT placetopright(TRECT r,int x,int y)
{
r.w=x-r.x;
r.h+=r.y-y;
r.y=y;
return r;
}
inline TRECT placebottomleft(TRECT r,int x,int y)
{
r.w+=r.x-x;
r.x=x;
r.h=y-r.y;
return r;
}
inline TRECT placebottomright(TRECT r,int x,int y)
{
r.w=x-r.x;
r.h=y-r.y;
return r;
}

//Convert between TTWOPOINT and TRECT.
inline TTWOPOINT rect2twopoint(TRECT r)
{
TTWOPOINT ret={r.x,r.y,r.x+r.w,r.y+r.h};
return ret;
}
inline TRECT twopoint2rect(TTWOPOINT tp)
{
TRECT ret;
if(tp.x1>tp.x2){
	ret.x=tp.x2;
	ret.w=tp.x1-tp.x2;
}else{
	ret.x=tp.x1;
	ret.w=tp.x2-tp.x1;
}
if(tp.y1>tp.y2){
	ret.y=tp.y2;
	ret.h=tp.y1-tp.y2;
}else{
	ret.y=tp.y1;
	ret.h=tp.y2-tp.y1;
}
return ret;
}

//Return TRECT, TTWOPOINT and TPOINT from values.
inline TRECT v2rect(int x,int y, int w, int h)
{
TRECT ret={x,y,w,h};
return ret;
}
inline TTWOPOINT v2twopoint(int x1,int y1, int x2, int y2)
{
TTWOPOINT ret={x1,y1,x2,y2};
return ret;
}
inline TPOINT v2point(int x,int y)
{
TPOINT ret={x,y};
return ret;
}
//Return TRECT and TTWOPOINT from TPOINTs.
inline TRECT points2rect(TPOINT p1,TPOINT p2)
{
TRECT ret;
if(p1.x>p2.x){
	ret.x=p2.x;
	ret.w=p1.x-p2.x;
}else{
	ret.x=p1.x;
	ret.w=p2.x-p1.x;
}
if(p1.y>p2.y){
	ret.y=p2.y;
	ret.h=p1.y-p2.y;
}else{
	ret.y=p1.y;
	ret.h=p2.y-p1.y;
}
return ret;
}
inline TTWOPOINT points2twopoint(TPOINT p1,TPOINT p2)
{
TTWOPOINT ret={p1.x,p1.y,p2.x,p2.y};
return ret;
}
inline TPOINT twopoint_first(TTWOPOINT tp)
{
TPOINT ret={tp.x1, tp.y1};
return ret;
}
inline TPOINT twopoint_second(TTWOPOINT tp)
{
TPOINT ret={tp.x2, tp.y2};
return ret;
}

//Return corners of a TRECT.
inline TPOINT recttopleft(TRECT r)
{
TPOINT ret={r.x,r.y};
return ret;
}
inline TPOINT recttopright(TRECT r)
{
TPOINT ret={r.x+r.w,r.y};
return ret;
}
inline TPOINT rectbottomleft(TRECT r)
{
TPOINT ret={r.x,r.y+r.h};
return ret;
}
inline TPOINT rectbottomright(TRECT r)
{
TPOINT ret={r.x+r.w,r.y+r.h};
return ret;
}

//Adds a point to another.
inline TPOINT addpoints(TPOINT point1, TPOINT point2)
{
point1.x+=point2.x;
point1.y+=point2.y;
return point1;
}
inline TPOINT subtractpoints(TPOINT point1,TPOINT point2)
{
point1.x-=point2.x;
point1.y-=point2.y;
return point1;
}
//Returns the relative position of the point in the rectangle.
inline TPOINT pointposinrect(TPOINT p,TRECT r)
{
p.x-=r.x;
p.y-=r.y;
return p;
}



/*****************
****          ****
**** add_clip ****
****          ****
******************
Changes clipping on a bimap. The area that the old clipping and the parameter
recangle has in common will be the new clipping. This automatically turns on
clipping.
*/
void add_clip(BITMAP *bmp,TRECT newclip)
{
if(bmp->clip)
	set_clip_fromrect(bmp,bothfitin(newclip,bothfitin(get_clip_asrect(bmp),get_size(bmp))));
else
	set_clip_fromrect(bmp,bothfitin(newclip,get_size(bmp)));
bmp->clip=TRUE;
}



/****************
****         ****
**** inrange ****
****         ****
*****************
Returns `TRUE' if the value is between min and max (inclusive).
*/
inline int inrange(int num,int min,int max)
{
return _inrange(num,min,max);
}
inline int intrange(int num,TRANGE r)
{
return _intrange(num,r);
}



/*******************************
****                        ****
**** nearest_pointposinrect ****
****                        ****
********************************
If the point is inside the rectangle, the point is returned. Otherwise the
point is moved to the nearest position that is inside the rectangle.
NOT TESTED!!!
*/
TPOINT nearest_pointposinrect(TPOINT p, TRECT r)
{
if(p.x<r.x)
	p.x=r.x;
else if(p.x>=(r.x+r.w))
	p.x=r.x+r.w-1;
if(p.y<r.y)
	p.y=r.y;
else if(p.y>=(r.y+r.h))
	p.y=r.y+r.h-1;
return p;
}



/**************
****       ****
**** dist2 ****
****       ****
***************
Returns the square of the distance between the two points.
*/
inline ulong dist2(TPOINT p1, TPOINT p2)
{
p1.x -= p2.x;
p1.y -= p2.y;
return p1.x * p1.x + p1.y * p1.y;
}
/*************
****      ****
**** dist ****
****      ****
**************
Returns the distance between the two points.
*/
inline ulong dist(TPOINT p1, TPOINT p2)
{
p1.x -= p2.x;
p1.y -= p2.y;
return sqrt(p1.x * p1.x + p1.y * p1.y);
}
/******************
****           ****
**** origodist ****
****           ****
*******************
Returns the distance from `p' to origo.
*/
inline ulong origodist(TPOINT p)
{
return sqrt(_sq(p.x) + _sq(p.y));
}
/*******************
****            ****
**** origodist2 ****
****            ****
********************
Returns the square of distance from `p' to origo.
*/
inline ulong origodist2(TPOINT p)
{
return _sq(p.x) + _sq(p.y);
}
/*****************
****          ****
**** dist2_2p ****
****          ****
******************
Returns the square of the distance between the two points.
*/
inline ulong dist2_2p(TTWOPOINT p)
{
p.x1 -= p.x2;
p.y1 -= p.y2;
return p.x1 * p.x1 + p.y1 * p.y1;
}
/****************
****         ****
**** dist_2p ****
****         ****
*****************
Returns the distance between the two points.
*/
inline ulong dist_2p(TTWOPOINT p)
{
p.x1 -= p.x2;
p.y1 -= p.y2;
return sqrt(p.x1 * p.x1 + p.y1 * p.y1);
}



/*******************************
****                        ****
**** rotate_point_direction ****
****                        ****
********************************
Rotates `p' around `centre' with the angle of `direction'.
*/
TPOINT rotate_point_direction(TPOINT p, TPOINT centre, TPOINT direction)
{
double distance;
//Avoid div by zero.
if(!(direction.x|direction.y)){
	errno = EDOM;
	return centre;
}
distance = _origodist(direction);
//`centre' is used to store the result in.
p.x -= centre.x;
p.y -= centre.y;
centre.x += d2l(((double)(p.x*direction.x - p.y*direction.y)) / distance);
centre.y += d2l(((double)(p.x*direction.y + p.y*direction.x)) / distance);
return centre;
}
/*********************
****              ****
**** rotate_point ****
****              ****
**********************
Rotates a point `p' around the axis `centre' `rotate_angle' radians.
*/
TPOINT rotate_point(TPOINT p, TPOINT centre, double rotate_angle)
{
double radius;
double angle;

p.x -= centre.x;
p.y -= centre.y;

radius = _origodist(p);
angle = atan2(p.y, p.x);

angle += rotate_angle;

centre.x += d2l(cos(angle) * radius);
centre.y += d2l(sin(angle) * radius);

return centre;
}
//Rotates a point `p' around the axis `centre' `rotate_angle'
//"Allegro-degrees" (256 degree units in a circle).
TPOINT frotate_point(TPOINT p, TPOINT centre, fixed rotate_angle)
{
fixed radius;
fixed angle;

p.x -= centre.x;
p.y -= centre.y;

radius = fsqrt(itofix(_sq(p.x) + _sq(p.y)));
angle = fatan2(itofix(p.y), itofix(p.x));

angle += rotate_angle;

centre.x += fcos(angle) * radius;
centre.y += fsin(angle) * radius;

return centre;
}






/***********************
****                ****
**** between_points ****
****                ****
************************
Returns a point that is part of the way between two points, at `weight'/256
position. (If `weight' is 0, `p1' is returned. If `weight' is 256, `p2' is
returned. If `weight' is 150, an angle that is somewhat nearer `p2' than `p1'
is returned.)
*/
TPOINT between_points(TPOINT p1, TPOINT p2, int weight)
{
p1.x += ((p2.x - p1.x) * weight + 128) >> 8;
p1.y += ((p2.y - p1.y) * weight + 128) >> 8;
return p1;
}



#endif
