/*
     _______                     ___                          ________
    /       \         /\        |   |\             /\        |        \
   /         >       /  \       |   ||            /  \       |         \
  /   ______/ >     /    \      |   ||           /    \      |    __    \
 <   <_______/     /      \     |   ||          /      \     |   |\_\    \
  \        \      /   /\   \    |   ||         /   /\   \    |   ||  \    \
   \        \    |   /_L\   |   |   ||        |   /_L\   |   |   ||   >   |\
    \_____   \   |          |\  |   ||        |          |\  |   ||  /    /|
   __L____>   >  |          ||  |   |L____    |          ||  |   |L_/    / /
  /          / > |   ____   ||  |         |\  |   ____   ||  |          / /
 <          / /  |   |\_|   ||  |         ||  |   |\_|   ||  |         / /
  \________/ /   |___|| |___||  |_________||  |___|| |___||  |________/ /
   \________/     \___\  \___\   \_________\   \___\  \___\   \_______\/


                an Addon Package for Allegro by Sven Sandberg


This file contains all functions for rectangles and points that you could
think of.

*/
#ifndef s_rect_h
#define s_rect_h

//For `ulong' etc.
#include "s_defs.h"
//For `_sq()'.
#include "math.h"
//For `BITMAP'.
#include <allegro.h>


/********************
 First the structs...
 ********************/
//Coordinates for a rectangle.
typedef struct{
	slong x,y;
	slong w,h;
}TRECT;
//Two points defining a rectangle. The functions don't use this - but you
//can convert between TRECT and TTWOPOINT.
typedef struct{
	slong x1,y1;
	slong x2,y2;
}TTWOPOINT;
//Coordinates for a point.
typedef struct{
	slong x,y;
}TPOINT;
//Coordinates for a 3d point.
typedef struct{
	slong x,y,z;
}T3DPOINT;
//Beginning and end of any range.
typedef struct{
	slong start,length;
}TRANGE;


//These are returned by some functions indicating no point or no rectangle.
extern TRECT norect;
extern TPOINT nopoint;
extern T3DPOINT no3dpoint;
extern TTWOPOINT notwopoint;
extern TRANGE norange;

extern TRECT zerorect;
extern TPOINT zeropoint;
extern T3DPOINT zero3dpoint;
extern TTWOPOINT zerotwopoint;
extern TRANGE zerorange;


/***********************************
 ...and then all the cool functions.
 ***********************************/
//Gets/sets clipping for bitmaps.
extern void set_clip_fromrect(BITMAP *bmp,TRECT rect);
extern TRECT get_clip_asrect(BITMAP *bmp);
//The area that newclip and old clipping has in common will be the new
//clipping for the bitmap.
extern void add_clip(BITMAP *bmp,TRECT newclip);
//Returns size of bitmap.
extern TRECT get_size(BITMAP *bmp);
//Sets clipping for the bitmap to its width and height.
extern void reset_clip(BITMAP *bmp);

//Returns `TRUE' if number is between min and max (inclusive).
#define _inrange(num,min,max) (((num)>=(min)) && ((num)<=(max)))
__INLINE__ int inrange(int num,int min,int max);
#define _intrange(num,r)      ( ((num)>=(r).start) && ((num)<=((r).start+(r).length)) )
__INLINE__ int intrange(int num,TRANGE r);

//Returns `TRUE' if rects intersect.
extern int intersect(TRECT r1,TRECT r2);
//Returns `TRUE' if point is inside rect.
extern int pointinrect(TPOINT p,TRECT r);
#define _pointinrect(p,r) (((p).x >= (r).x)&&((p).x < ((r).x+(r).w)) && ((p).y >= (r).y)&&((p).y < ((r).y+(r).h)))
//Returns a rectangle in which both the parameter rectangles will fit.
extern TRECT bothfitin(TRECT r1,TRECT r2);
//Returns a rectangle that will fit in both parameter rectangles, i.e. the
//intersection of them. If they don't intersect, a 0,0,0,0-rectangle is
//returned.
extern TRECT fitinboth(TRECT r1,TRECT r2);
//Returns a point between two points.
extern TPOINT midpoint(TPOINT p1,TPOINT p2);
//Changes the size of a TRECT. The center of the rectangle remains on the
//same place.
extern TRECT resizerect(TRECT r,int bypixels);
//Move one side or corner of a rectangle relative to old position.
extern inline TRECT moveleft(TRECT r,int x);
extern inline TRECT movetop(TRECT r,int y);
extern inline TRECT moveright(TRECT r,int x);
extern inline TRECT movebottom(TRECT r,int y);
extern inline TRECT movetopleft(TRECT r,int x,int y);
extern inline TRECT movetopright(TRECT r,int x, int y);
extern inline TRECT movebottomleft(TRECT r,int x, int y);
extern inline TRECT movebottomright(TRECT r,int x, int y);
//Place one side or corner of a recangle at an absolute position.
extern inline TRECT placeleft(TRECT r,int x);
extern inline TRECT placetop(TRECT r,int y);
extern inline TRECT placeright(TRECT r,int x);
extern inline TRECT placebottom(TRECT r,int y);
extern inline TRECT placetopleft(TRECT r,int x,int y);
extern inline TRECT placetopright(TRECT r,int x,int y);
extern inline TRECT placebottomleft(TRECT r,int x,int y);
extern inline TRECT placebottomright(TRECT r,int x,int y);
extern inline TRECT v2rect(int x,int y, int w, int h);
//Return the struct from its values.
extern inline TTWOPOINT v2twopoint(int x1,int y1, int x2, int y2);
extern inline TPOINT v2point(int x,int y);
extern inline TTWOPOINT rect2twopoint(TRECT r);
//Convert between TTWOPOINT, TRECT and TPOINT.
extern inline TRECT twopoint2rect(TTWOPOINT tp);
extern inline TPOINT twopoint_first(TTWOPOINT tp);
extern inline TPOINT twopoint_second(TTWOPOINT tp);
extern inline TRECT points2rect(TPOINT p1,TPOINT p2);
extern inline TTWOPOINT points2twopoint(TPOINT p1,TPOINT p2);
extern inline TPOINT recttopleft(TRECT rect);
extern inline TPOINT recttopright(TRECT rect);
extern inline TPOINT rectbottomleft(TRECT rect);
extern inline TPOINT rectbottomright(TRECT rect);
//Macros that could be used inside parameter lists that extract the
//coordinates of a TRECT.
#define _xrect(r)    (r).x, (r).y, (r).w, (r).h
#define _xabsrect(r) (r).x, (r).y, (r).x + (r).w, (r).y + (r).h

//Returns a point that is at the relative position point2 from point1.
extern inline TPOINT addpoints(TPOINT point1, TPOINT point2);
//Returns the relative position of point1 from point2.
extern inline TPOINT subtractpoints(TPOINT point1,TPOINT point2);
//Returns the relative position of the point to the rect.
extern inline TPOINT pointposinrect(TPOINT p,TRECT r);
//Returns the point in the rectangle that is neares to the point p.
extern TPOINT nearest_pointposinrect(TPOINT p, TRECT r);

#define _dist(p1, p2)   (sqrt(_sq(p1.x - p2.x) + _sq(p1.y - p2.y)))
#define _dist2(p1, p2)  (_sq(p1.x - p2.x) + _sq(p1.y - p2.y))
#define _origodist(p)   (sqrt(_sq(p.x) + _sq(p.y)))
#define _origodist2(p)  (_sq(p.x) + _sq(p.y))
//Return the distance between two points.
extern inline ulong dist(TPOINT p1, TPOINT p2);
extern inline ulong dist_2p(TTWOPOINT p);
extern inline ulong origodist(TPOINT p);
//Return the square of the distance between two points.
extern inline ulong dist2(TPOINT p1, TPOINT p2);
extern inline ulong dist2_2p(TTWOPOINT p);
extern inline ulong origodist2(TPOINT p);

//Rotates the point `p'.
extern TPOINT rotate_point_direction(TPOINT p, TPOINT centre,
 TPOINT direction);

//Rotates the point `p' around `centre' the angle `rotate_angle'.
extern TPOINT rotate_point(TPOINT p, TPOINT centre, double rotate_angle);
extern TPOINT frotate_point(TPOINT p, TPOINT centre, fixed rotate_angle);

//Returns a point that is part of the way between two points.
extern TPOINT between_points(TPOINT p1, TPOINT p2, int weight);

#endif
