/* db.c: an external database to avoid filesystem lookups.

Copyright (C) 1994 Karl Berry.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <kpathsea/config.h>
#include <kpathsea/c-fopen.h>
#include <kpathsea/c-pathch.h>
#include <kpathsea/db.h>
#include <kpathsea/hash.h>
#include <kpathsea/line.h>
#include <kpathsea/readable.h>
#include <kpathsea/str-list.h>


/* Don't bother implementing a search path for this until people ask.  */
#ifndef DB_FILE
#define DB_FILE "/usr/local/lib/texmf/ls-R"
#endif

/* If no DB_FILE, do nothing.  Maybe they aren't using this feature.  */

static void
db_build P1C(hash_table_type *, table)
{
  string line;
  string cur_dir = NULL; /* First thing in ls-R might be a filename.  */
  FILE *db_file = fopen (DB_FILE, FOPEN_R_MODE);
  if (!db_file)
    return;
  
  while ((line = read_line (db_file)) != NULL)
    {
      unsigned len = strlen (line);
      if (line[0] == DIR_SEP && line[len - 1] == ':') /* /...: => new dir */
        {
          cur_dir = xstrdup (line);
          cur_dir[len - 1] = DIR_SEP;
        }
      else if (line[0] != 0 && cur_dir) /* other nonblank line */
        { /* New hash table entry with a key of `line' and a data of
             `cur_dir'.  Already-existing identical keys are ok, since
             a file named `foo' can be in more than one directory.
             Since it doesn't hurt, share the directory name string
             among all the files in the directory. */
          hash_insert (table, xstrdup (line), cur_dir);
        }
      /* else ignore blank lines */
      
      free (line);
    }
  
  xfclose (db_file, DB_FILE);
}

/* Return true if FILENAME could be in PATH_ELT, i.e., if the directory
   part of FILENAME matches PATH_ELT.  Have to consider // wildcards, but
   $ and ~ expansion have already been done.  */
     
static boolean
match P2C(const_string, filename,  const_string, path_elt)
{
  return true; /* Punt for now.  */
}

str_list_type
kpse_db_search P3C(const_string, name,  const_string, path_elt,  boolean, all)
{
  static hash_table_type db;
  str_list_type ret;
  string *db_dirs;
  boolean done = false;
  
  /* Hash up the database if this is the first call.  */
  if (db.size == 0)
    {
      db = hash_create (7603); /* What the heck, sparse is ok.  */
      db_build (&db);
    }
  
  /* Look up NAME.  */
  db_dirs = hash_lookup (db, name);

  ret = str_list_init (); /* Some old compilers ...  */
  
  /* For each filename found, see if it matches the path element.  For
     example, if we have ../cx/cmr10.300pk and .../ricoh/cmr10.300pk,
     and the path looks like .../cx, we don't want the ricoh file.  */
  while (!done && db_dirs && *db_dirs)
    {
      string db_file = concat (*db_dirs, name);
      
      if (match (db_file, path_elt) && kpse_readable_file (db_file))
        {
          str_list_add (&ret, db_file);
          if (!all) done = true;
        }
      else
        free (db_file);
      
      /* On to the next directory, if any.  */
      db_dirs++;
    }
  
  /* This is just the space for the pointers, not the strings.  */
  if (db_dirs && *db_dirs) free (db_dirs);
  
  return ret;
}
